-- Get core objects
local Gladdy = LibStub("AceAddon-3.0"):GetAddon("Gladdy")
local L = LibStub("AceLocale-3.0"):GetLocale("Gladdy")

-- Shared media
local LSM = LibStub("LibSharedMedia-3.0")
local AceGUIWidgetLSMlists = AceGUIWidgetLSMlists
LSM:Register("statusbar", "Minimalist", "Interface\\AddOns\\Gladdy\\statusbar\\Minimalist")

-- Lua calls
local pairs = pairs
local tinsert = table.insert
local tonumber = tonumber
local tostring = tostring

-- Blizzard API
local MACRO, TARGET, FOCUS, ADDON_DISABLED = MACRO, TARGET, FOCUS, ADDON_DISABLED

-- Default options
Gladdy.defaults = {
    profile = {
        x = 0,
        y = 0,
        locked = false,
        growUp = false,
        frameScale = 1,
        padding = 5,
        frameColor = { r = 0, g = 0, b = 0, a = .4 },
        highlight = true,
        targetBorder = true,
        focusBorder = true,
        leaderBorder = true,
        trinketStatus = false,
        announceType = "rw",
        enemyAnnounce = false,
        specAnnounce = false,
        drinkAnnounce = true,
        resAnnounce = true,
        auraAnnounce = false,
        trinketUpAnnounce = false,
        lowHealthAnnounce = true,
        lowHealthPercentage = 20,
        castBar = true,
        powerBar = true,
        barWidth = 180,
        barHeight = 25,
        manaBarHeight = 15,
        castBarHeight = 20,
        barBottomMargin = 10,
        barTexture = "Minimalist",
        targetBorderColor = { r = 1, g = .7, b = 0, a = 1 },
        focusBorderColor = { r = 1, g = 0, b = 0, a = 1 },
        leaderBorderColor = { r = 0, g = 1, b = 0, a = 1 },
        manaColor = { r = .18, g = .44, b = .75, a = 1 },
        manaDefault = true,
        energyColor = { r = 1, g = 1, b = 0, a = 1 },
        energyDefault = true,
        rageColor = { r = 1, g = 0, b = 0, a = 1 },
        rageDefault = true,
        castBarColor = { r = 1, g = 1, b = 0, a = 1 },
        castBarBgColor = { r = 1, g = 1, b = 1, a = .3 },
        shortHpMana = true,
        healthPercentage = true,
        healthActual = false,
        healthMax = true,
        manaText = true,
        manaPercentage = true,
        manaActual = false,
        manaMax = true,
        raceText = true,
        specText = true,
        drText = true,
        healthFontSize = 11,
        manaFontSize = 10,
        castBarFontSize = 9,
        auraFontSize = 16,
        drFontSize = 11,
        healthFontColor = { r = 1, g = 1, b = 1, a = 1 },
        manaFontColor = { r = 1, g = 1, b = 1, a = 1 },
        castBarFontColor = { r = 1, g = 1, b = 1, a = 1 },
        auraFontColor = { r = 0, g = 1, b = 0, a = 1 },
        drFontColor = { r = 0, g = 1, b = 0, a = 1 },
        drCooldown = true,
        drCooldownPos = "LEFT",
        drIconAdjust = false,
        drIconSize = 30,
        drCooldownAnchor = "TOP",
        auraList = {}, -- Announce list
        auras = {},
        drList = {},
        attributes = {
            { name = "Target", button = "1", modifier = "", action = "target", spell = "" },
            { name = "Focus", button = "2", modifier = "", action = "focus", spell = "" },
        },
    },
}

-- Update auralist
for k, v in pairs(Gladdy._AURAS_) do
    if (v.track) then
        tinsert(Gladdy.defaults.profile.auras, {
            name = k,
            priority = v.priority or 0,
            deleted = false,
        })
    end
end

-- Update attributes
for i = 3, 10 do
    tinsert(Gladdy.defaults.profile.attributes, {
        name = L["Action #%d"]:format(i),
        button = "",
        modifier = "",
        action = "disabled",
        spell = "",
    })
end

--- SlashHandler
-- Gladdy slash handler, used by AceConsole
local SlashHandler = function(msg)
    if (msg:find("test")) then
        local test = tonumber(msg:match("^test(.+)"))
        
        if (not test or test > 5 or test < 2 or test == 4) then
            test = 5
        end
        
        Gladdy:ToggleFrame(test)
    elseif (msg == "ui" or msg == "config" or msg == "options") then
        Gladdy:ShowOptions()
    elseif (msg == "hide") then
        Gladdy:HideFrame()
    else
        Gladdy:Print(L["Valid slash commands are:"])
        Gladdy:Print("/gladdy ui")
        Gladdy:Print("/gladdy test1-5")
        Gladdy:Print("/gladdy hide")
    end
end

--- getOption
-- Default option getter
local function getOption(info)
    local key = info.arg or info[#(info)]
    return Gladdy.db[key]
end

--- setOption
-- Default option setter
local function setOption(info, value)
    local key = info.arg or info[#(info)]
    Gladdy.db[key] = value
    Gladdy:UpdateFrame()
end

--- getColorOption
-- Color option getter
local function getColorOption(info)
    local key = info.arg or info[#(info)]
    return Gladdy.db[key].r, Gladdy.db[key].g, Gladdy.db[key].b, Gladdy.db[key].a
end

--- setColorOption
-- Color option setter
local function setColorOption(info, r, g, b, a)
    local key = info.arg or info[#(info)]
    Gladdy.db[key].r, Gladdy.db[key].g, Gladdy.db[key].b, Gladdy.db[key].a = r, g, b ,a
    Gladdy:UpdateFrame()
end

--- getAura
-- Aura option getter
local function getAura(info)
    return Gladdy.db.auras[tonumber(info[#(info) - 1])][info[#(info)]]
end

--- setAura
-- Aura option setter
local function setAura(info, value)
    Gladdy.db.auras[tonumber(info[#(info) - 1])][info[#(info)]] = value
    
    -- Update name in options
    if (info[#(info)] == "name") then
        Gladdy.options.args.auras.args[info[#(info) - 1]].name = value
    end
    
    Gladdy:ConvertAuraList()
    Gladdy:UpdateFrame()
end

--- getAttribute
-- Attribute option getter
local function getAttribute(info)
    return Gladdy.db.attributes[tonumber(info[#(info) - 1])][info[#(info)]]
end

--- setAttribute
-- Attribute option setter
local function setAttribute(info, value)
    Gladdy.db.attributes[tonumber(info[#(info) - 1])][info[#(info)]] = value
    
    -- Update name in options
    if (info[#(info)] == "name") then
        Gladdy.options.args.clicks.args[info[#(info) - 1]].name = value
    end
    
    Gladdy:UpdateFrame()
end

-- Modifier list
local modifiers = { [""] = L["None"], ["ctrl-"] = L["CTRL"], ["shift-"] = L["SHIFT"], ["alt-"] = L["ALT"] }
-- Button list
local buttons = { ["1"] = L["Left button"], ["2"] = L["Right button"], ["3"] = L["Middle button"], ["4"] = L["Button 4"], ["5"] = L["Button 5"] }
-- Click list
local clicks = { ["macro"] = MACRO, ["target"] = TARGET, ["focus"] = FOCUS, ["spell"] = L["Cast Spell"], ["disabled"] = ADDON_DISABLED }

-- Aura announce list
local auraAnnounces = {
    ["disabled"] = ADDON_DISABLED,
    ["self"] = L["Self"],
    ["party"] = L["Party"],
    ["bg"] = L["Battleground"],
    ["rw"] = L["Raid Warning"],
    ["sct"] = L["Scrolling Combat Text"],
    ["msbt"] = L["MikScrollingBattleText"],
    ["fct"] = L["Blizzard's Floating Combat Text"],
    ["parrot"] = L["Parrot"],
}

-- Common announce list
local announces = {
    ["self"] = L["Self"],
    ["party"] = L["Party"],
    ["bg"] = L["Battleground"],
    ["rw"] = L["Raid Warning"],
    ["sct"] = L["Scrolling Combat Text"],
    ["msbt"] = L["MikScrollingBattleText"],
    ["fct"] = L["Blizzard's Floating Combat Text"],
    ["parrot"] = L["Parrot"],
}

--- SetupAuraOption
-- Create new aura option
local function SetupAuraOption(number)
    return {
        type = "group",
        name = Gladdy.db.auras[number].name,
        desc = Gladdy.db.auras[number].name,
        order = number,
        get = getAura,
        set = setAura,
        args = {
            name = {
                type = "input",
                name = L["Aura name"],
                desc = L["Name of the aura"],
                order = 1,
            },
            priority = {
                type = "range",
                name = L["Priority"],
                desc = L["Select what priority the aura should have"],
                order = 2,
                min = 0,
                max = 50,
                step = 5,
            },
            announce = {
                type = "select",
                name = L["Announcement type"],
                desc = L["Where to display the announcement messages"],
                order = 3,
                values = auraAnnounces,
                get = function(info)
                    local name = Gladdy.options.args.auras.args[info[#(info) - 1]].name
                    return Gladdy.db.auraList[name] or "disabled"
                end,
                set = function(info, value)
                    local name = Gladdy.options.args.auras.args[info[#(info) - 1]].name
                    Gladdy.db.auraList[name] = value
                    
                    Gladdy:ConvertAuraList()
                    Gladdy:UpdateFrame()
                end,
            },
            delete = {
                type = "execute",
                name = L["Delete"],
                desc = L["Delete aura"],
                order = 4,
                func = function(info)
                    local key = tonumber(info[#(info) - 1])
                    local name = Gladdy.db.auras[key].name
                    
                    Gladdy.db.auraList[name] = nil
                    Gladdy.db.auras[key].deleted = true
                    Gladdy.options.args.auras.args = {}
                    
                    for i = #(Gladdy.db.auras), 1, -1 do
                        if (not Gladdy.db.auras[i].deleted) then
                            Gladdy.options.args.auras.args[tostring(i)] = SetupAuraOption(i)
                        end
                    end
                    
                    Gladdy:ConvertAuraList()
                end,
            },
        },
    }
end

--- SetupAttributeOption
-- Create new attribute option
local function SetupAttributeOption(number)
    return {
        type = "group",
        name = Gladdy.db.attributes[number].name,
        desc = Gladdy.db.attributes[number].name,
        order = number,
        get = getAttribute,
        set = setAttribute,
        args = {
            name = {
                type = "input",
                name = L["Name"],
                desc = L["Select the name of the click option"],
                order = 1,
            },
            button = {
                type = "select",
                name = L["Button"],
                desc = L["Select which mouse button to use"],
                order = 2,
                values = buttons,
            },
            modifier = {
                type = "select",
                name = L["Modifier"],
                desc = L["Select which modifier to use"],
                order = 3,
                values = modifiers,
            },
            action = {
                type = "select",
                name = L["Action"],
                desc = L["Select what action this mouse button does"],
                order = 4,
                values = clicks,
            },
            spell = {
                type = "input",
                name = L["Spell name / Macro text"],
                desc = L["Use *name* as unit's name. Like a '/rofl *name*'"],
                order = 5,
                multiline = true,
            },
        },
    }
end

--- Gladdy:SetupOptions
-- Initialize options
function Gladdy:SetupOptions()   
    -- Init options
    self.options = {
        type = "group",
        name = "Gladdy",
        icon = "Interface\\PVPFrame\\PVP-ArenaPoints-Icon",
        plugins = {},
        get = getOption,
        set = setOption,
        args = {
            general = {
                type = "group",
                name = L["General"],
                desc = L["General settings"],
                order = 1,
                args = {
                    locked = {
                        type = "toggle",
                        name = L["Lock frame"],
                        desc = L["Toggle if frame can be moved"],
                        order = 1,
                    },
                    growUp = {
                        type = "toggle",
                        name = L["Grow frame upwards"],
                        desc = L["If enabled the frame will grow upwards instead of downwards"],
                        order = 2,
                    },
                    frameScale = {
                        type = "range",
                        name = L["Frame scale"],
                        desc = L["Scale of the frame"],
                        order = 3,
                        min = .1,
                        max = 2,
                        step = .1,
                    },
                    padding = {
                        type = "range",
                        name = L["Frame padding"],
                        desc = L["Padding of the frame"],
                        order = 4,
                        min = 0,
                        max = 20,
                        step = 1,
                    },
                    frameColor = {
                        type = "color",
                        name = L["Frame color"],
                        desc = L["Color of the frame"],
                        order = 5,
                        hasAlpha = true,
                        get = getColorOption,
                        set = setColorOption,
                    },
                    highlight = {
                        type = "toggle",
                        name = L["Highlight target"],
                        desc = L["Toggle if the selected target should be highlighted"],
                        order = 6,
                    },
                    targetBorder = {
                        type = "toggle",
                        name = L["Show border around target"],
                        desc = L["Toggle if a border should be shown around the selected target"],
                        order = 7,
                    },
                    focusBorder = {
                        type = "toggle",
                        name = L["Show border around focus"],
                        desc = L["Toggle of a border should be shown around the current focus"],
                        order = 8,
                    },
                    leaderBorder = {
                        type = "toggle",
                        name = L["Show border around party leader"],
                        desc = L["Toggle if a border should be shown around the party leader"],
                        order = 9,
                    },
                    trinketStatus = {
                        type = "toggle",
                        name = L["Show PvP trinket status"],
                        desc = L["Show PvP trinket status to the right of the enemy name"],
                        order = 10,
                    },
                    announcements = {
                        type = "group",
                        name = L["Announcements"],
                        desc = L["Set options for different announcements"],
                        order = 11,
                        args = {
                            announceType = {
                                type = "select",
                                name = L["Announce type"],
                                desc = L["How should we announce"],
                                order = 1,
                                values = announces,
                            },
                            enemyAnnounce = {
                                type = "toggle",
                                name = L["New enemies"],
                                desc = L["Announce new enemies found"],
                                order = 2,
                            },
                            specAnnounce = {
                                type = "toggle",
                                name = L["Talent spec detection"],
                                desc = L["Announce when an enemy's talent spec is discovered"],
                                order = 3,
                            },
                            drinkAnnounce = {
                                type = "toggle",
                                name = L["Drinking"],
                                desc = L["Announces enemies that start to drink"],
                                order = 4,
                            },
                            resAnnounce = {
                                type = "toggle",
                                name = L["Resurrections"],
                                desc = L["Announces enemies who starts to cast a resurrection spell"],
                                order = 5,
                            },
                            auraAnnounce = {
                                type = "toggle",
                                name = L["Aura"],
                                desc = L["Announces enemies who gain an aura"],
                                order = 6,
                            },
                            trinketUpAnnounce = {
                                type = "toggle",
                                name = L["Trinket ready"],
                                desc = L["Announce when an enemy's trinket is ready again"],
                                order = 7,
                            },
                            lowHealthAnnounce = {
                                type = "toggle",
                                name = L["Enemies on low health"],
                                desc = L["Announce enemies that go below a certain percentage of health"],
                                order = 8,
                            },
                            lowHealthPercentage = {
                                type = "range",
                                name = L["Low health percentage"],
                                desc = L["The percentage when enemies are counted as having low health"],
                                order = 9,
                                min = 1,
                                max = 100,
                                step = 1,
                                disabled = function() return not self.db.lowHealthAnnounce end,
                            },
                        },
                    },
                },
            },
            bars = {
                type = "group",
                name = L["Bars"],
                desc = L["Bars settings"],
                order = 2,
                args = {
                    castBar = {
                        type = "toggle",
                        name = L["Show cast bars"],
                        desc = L["Show cast bars"],
                        order = 1,
                    },
                    powerBar = {
                        type = "toggle",
                        name = L["Show power bars"],
                        desc = L["Show power bars"],
                        order = 2,
                    },
                    sizes = {
                        type = "group",
                        name = L["Size and margin"],
                        desc = L["Size and margin settings"],
                        order = 3,
                        args = {
                            barWidth = {
                                type = "range",
                                name = L["Bar width"],
                                desc = L["Width of the health/power bars"],
                                order = 1,
                                min = 20,
                                max = 500,
                                step = 5,
                            },
                            barHeight = {
                                type = "range",
                                name = L["Bar height"],
                                desc = L["Width of the health bar"],
                                order = 2,
                                min = 5,
                                max = 50,
                                step = 1,
                            },
                            manaBarHeight = {
                                type = "range",
                                name = L["Power bar height"],
                                desc = L["Height of the power bar"],
                                order = 3,
                                min = 5,
                                max = 20,
                                step = 1,
                                disabled = function() return not self.db.powerBar end,
                            },
                            castBarHeight = {
                                type = "range",
                                name = L["Cast bar height"],
                                desc = L["Height of the cast bar"],
                                order = 4,
                                min = 5,
                                max = 50,
                                step = 1,
                                disabled = function() return not self.db.castBar end,
                            },
                            barBottomMargin = {
                                type = "range",
                                name = L["Bar bottom margin"],
                                desc = L["Margin to the next bar"],
                                order = 5,
                                min = 0,
                                max = 30,
                                step = 1,
                            },
                        },
                    },
                    colors = {
                        type = "group",
                        name = L["Colors"],
                        desc = L["Color settings"],
                        order = 4,
                        args = {
                            barTexture = {
                                type = "select",
                                name = L["Bar texture"],
                                desc = L["Texture of health/cast bars"],
                                order = 1,
                                dialogControl = "LSM30_Statusbar",
                                values = AceGUIWidgetLSMlists.statusbar,
                            },
                            targetBorderColor = {
                                type = "color",
                                name = L["Target border color"],
                                desc = L["Color of the target's border"],
                                order = 2,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                            },
                            focusBorderColor = {
                                type = "color",
                                name = L["Focus border color"],
                                desc = L["Color of the focus border"],
                                order = 3,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                            },
                            leaderBorderColor = {
                                type = "color",
                                name = L["Party leader border color"],
                                desc = L["Color of the party leader border"],
                                order = 4,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                            },
                            manaColor = {
                                type = "color",
                                name = L["Mana color"],
                                desc = L["Color of the mana bar"],
                                order = 5,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.powerBar or self.db.manaDefault end,
                            },
                            manaDefault = {
                                type = "toggle",
                                name = L["Game default"],
                                desc = L["Use game default mana color"],
                                order = 6,
                            },
                            energyColor = {
                                type = "color",
                                name = L["Energy color"],
                                desc = L["Color of the energy bar"],
                                order = 7,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.powerBar or self.db.energyDefault end,
                            },
                            energyDefault = {
                                type = "toggle",
                                name = L["Game default"],
                                desc = L["Use game default energy color"],
                                order = 8,
                            },
                            rageColor = {
                                type = "color",
                                name = L["Rage color"],
                                desc = L["Color of the rage bar"],
                                order = 9,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.powerBar or self.db.rageDefault end,
                            },
                            rageDefault = {
                                type = "toggle",
                                name = L["Game default"],
                                desc = L["Use game default rage color"],
                                order = 10,
                            },
                            castBarColor = {
                                type = "color",
                                name = L["Cast bar color"],
                                desc = L["Color of the cast bar"],
                                order = 11,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.castBar end,
                            },
                            castBarBgColor = {
                                type = "color",
                                name = L["Cast bar background color"],
                                desc = L["Color of the cast bar background"],
                                order = 12,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.castBar end,
                            },
                        },
                    },
                },
            },
            text = {
                type = "group",
                name = L["Text"],
                desc = L["Text settings"],
                order = 3,
                args = {
                    shortHpMana = {
                        type = "toggle",
                        name = L["Shorten Health/Power text"],
                        desc = L["Shorten the formatting of the health and power text to e.g. 20.0/25.3 when the amount is over 9999"],
                        order = 1,
                    },
                    healthPercentage = {
                        type = "toggle",
                        name = L["Show health percentage"],
                        desc = L["Show health percentage on the health bar"],
                        order = 2,
                    },
                    healthActual = {
                        type = "toggle",
                        name = L["Show the actual health"],
                        desc = L["Show the actual health on the health bar"],
                        order = 3,
                    },
                    healthMax = {
                        type = "toggle",
                        name = L["Show max health"],
                        desc = L["Show maximum health on the health bar"],
                        order = 4,
                    },
                    manaText = {
                        type = "toggle",
                        name = L["Show power text"],
                        desc = L["Show mana/energy/rage text on the power bar"],
                        order = 5,
                        disabled = function() return not self.db.powerBar end,
                    },
                    manaPercentage = {
                        type = "toggle",
                        name = L["Show power percentage"],
                        desc = L["Show mana/energy/rage percentage on the power bar"],
                        order = 6,
                        disabled = function() return not self.db.powerBar or not self.db.manaText end,
                    },
                    manaActual = {
                        type = "toggle",
                        name = L["Show the actual power"],
                        desc = L["Show the actual mana/energy/rage on the power bar"],
                        order = 7,
                        disabled = function() return not self.db.powerBar or not self.db.manaText end,
                    },
                    manaMax = {
                        type = "toggle",
                        name = L["Show max power"],
                        desc = L["Show maximum mana/energy/rage on the power bar"],
                        order = 8,
                        disabled = function() return not self.db.powerBar or not self.db.manaText end,
                    },
                    raceText = {
                        type = "toggle",
                        name = L["Show race text"],
                        desc = L["Show race text on the power bar"],
                        order = 9,
                        disabled = function() return not self.db.powerBar end,
                    },
                    specText = {
                        type = "toggle",
                        name = L["Show spec text"],
                        desc = L["Show spec text on the power bar"],
                        order = 10,
                        disabled = function() return not self.db.powerBar end,
                    },
                    drText = {
                        type = "toggle",
                        name = L["Show DR text"],
                        desc = L["Show DR text on the icons"],
                        order = 11,
                        disabled = function() return not self.db.drCooldown end,
                    },
                    sizes = {
                        type = "group",
                        name = L["Sizes"],
                        desc = L["Size settings"],
                        order = 12,
                        args = {
                            healthFontSize = {
                                type = "range",
                                name = L["Health text size"],
                                desc = L["Size of the health bar text"],
                                order = 1,
                                min = 1,
                                max = 20,
                                step = 1,
                            },
                            manaFontSize = {
                                type = "range",
                                name = L["Mana text size"],
                                desc = L["Size of the mana bar text"],
                                order = 2,
                                min = 1,
                                max = 20,
                                step = 1,
                                disabled = function() return not self.db.powerBar end,
                            },
                            castBarFontSize = {
                                type = "range",
                                name = L["Cast bar text size"],
                                desc = L["Size of the cast bar text"],
                                order = 3,
                                min = 1,
                                max = 20,
                                step = 1,
                                disabled = function() return not self.db.castBar end,
                            },
                            auraFontSize = {
                                type = "range",
                                name = L["Aura text size"],
                                desc = L["Size of the aura text"],
                                order = 4,
                                min = 1,
                                max = 20,
                                step = 1,
                                disabled = function() return not self.db.auras end,
                            },
                            drFontSize = {
                                type = "range",
                                name = L["DR text size"],
                                desc = L["Size of the DR text"],
                                order = 5,
                                min = 1,
                                max = 20,
                                step = 1,
                                disabled = function() return not self.db.drCooldown end,
                            },
                        },
                    },
                    colors = {
                        type = "group",
                        name = L["Colors"],
                        desc = L["Color settings"],
                        order = 13,
                        args = {
                            healthFontColor = {
                                type = "color",
                                name = L["Health text color"],
                                desc = L["Color of the health bar text"],
                                order = 1,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                            },
                            manaFontColor = {
                                type = "color",
                                name = L["Mana text color"],
                                desc = L["Color of the mana bar text"],
                                order = 2,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.powerBar end,
                            },
                            castBarFontColor = {
                                type = "color",
                                name = L["Cast bar text color"],
                                desc = L["Color of the cast bar text"],
                                order = 3,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.castBar end,
                            },
                            auraFontColor = {
                                type = "color",
                                name = L["Aura text color"],
                                desc = L["Color of the aura text"],
                                order = 4,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.auras end,
                            },
                            drFontColor = {
                                type = "color",
                                name = L["DR text color"],
                                desc = L["Color of the DR text"],
                                order = 5,
                                hasAlpha = true,
                                get = getColorOption,
                                set = setColorOption,
                                disabled = function() return not self.db.drCooldown end,
                            },
                        },
                    },
                },
            },
            dr = {
                type = "group",
                name = L["DR tracker"],
                desc = L["DR settings"],
                order = 4,
                args = {
                    drCooldown = {
                        type = "toggle",
                        name = L["Show icons"],
                        desc = L["Show DR cooldown icons"],
                        order = 1,
                    },
                    drCooldownPos = {
                        type = "select",
                        name = "DR cooldown position",
                        desc = "Position of the DR icons",
                        order = 2,
                        values = {
                            ["LEFT"] = L["Left"],
                            ["RIGHT"] = L["Right"],
                        },
                    },
                    drIconAdjust = {
                        type = "toggle",
                        name = "Adjust Icon Size",
                        desc = "Adjust Icon Size on the Unit's Height",
                        order = 3,
                    },
                    drIconSize = {
                        type = "range",
                        name = L["Icon Size"],
                        desc = L["Size of the DR Icons"],
                        order = 4,
                        min = 5,
                        max = 100,
                        step = 1,
                        disabled = function() return self.db.drIconAdjust end,
                    },
                    drCooldownAnchor = {
                        type = "select",
                        name = L["DR Cooldown anchor"],
                        desc = L["Anchor of the cooldown icons"],
                        order = 5,
                        values = {
                            ["TOP"] = L["Top"],
                            ["CENTER"] = L["Center"],
                            ["BOTTOM"] = L["Bottom"],
                        },
                        disabled = function() return self.db.drIconAdjust end,
                    },
                },
            },
            auras = {
                type = "group",
                name = L["Auras"],
                desc = L["Aura settings"],
                order = 5,
                args = {},
            },
            clicks = {
                type = "group",
                name = L["Clicks"],
                desc = L["Click settings"],
                order = 6,
                args = {},
            },
        },
    }
    
    -- Populate auras
    for i = #(self.db.auras), 1, -1 do
        if (not self.db.auras[i].deleted) then
            self.options.args.auras.args[tostring(i)] = SetupAuraOption(i)
        end
    end
    
    -- Reload list
    self:ConvertAuraList()
    
    -- Populate clicks
    for i = 1, 10 do
        self.options.args.clicks.args[tostring(i)] = SetupAttributeOption(i)
    end
    
    -- Initialize profiles
    self.options.plugins.profiles = { profiles = LibStub("AceDBOptions-3.0"):GetOptionsTable(self.dbi) }
    
    -- Add to Blizzard options
    LibStub("AceConfig-3.0"):RegisterOptionsTable("Gladdy", self.options)
    LibStub("AceConfigDialog-3.0"):AddToBlizOptions("Gladdy", "Gladdy")
    
    -- Enable slash
    self:RegisterChatCommand("gladdy", SlashHandler)
end

--- Gladdy:ShowOptions
-- Show Gladdy options frame
function Gladdy:ShowOptions()
    InterfaceOptionsFrame_OpenToFrame("Gladdy")
end