-- Get core objects
local Gladdy = LibStub("AceAddon-3.0"):GetAddon("Gladdy")
local L = LibStub("AceLocale-3.0"):GetLocale("Gladdy")
local LSM = LibStub("LibSharedMedia-3.0")

-- Lua calls
local select = select

-- Blizzard API
local CreateFrame = CreateFrame
local UIParent = UIParent
local InCombatLockdown = InCombatLockdown
local RAID_CLASS_COLORS = RAID_CLASS_COLORS
local GetSpellInfo = GetSpellInfo

-- Local constants
local POWER_COLOR = {
    [0] = { r = .18, g = .44, b = .75, a = 1 },
    [1] = { r = 1, g = 0, b = 0, a = 1 },
    [3] = { r = 1, g = 1, b = 0, a = 1 },
}

--- AuraUpdate
-- Aura update handler
local function AuraUpdate(f, elapsed)
    if (f.active) then
        f.timeLeft = f.timeLeft - elapsed
        
        if (f.timeLeft <= 0) then
            Gladdy:AuraFades(f)
            return
        end
        
        f.text:SetFormattedText("%.1f", f.timeLeft)
    end
end

--- CastUpdate
-- Cast updater
local function CastUpdate(f, elapsed)
    if (f.isCasting) then
        if (f.value >= f.maxValue) then
            f:SetValue(f.maxValue)
            Gladdy:HandleCastEnd(f)
            return
        end
        f.value = f.value + elapsed
        f:SetValue(f.value)
        f.timeText:SetFormattedText("%.1f", f.maxValue - f.value)
    elseif (f.isChanneling) then
        if (f.value <= 0) then
            Gladdy:HandleCastEnd(f)
            return
        end
        f.value = f.value - elapsed
        f:SetValue(f.value)
        f.timeText:SetFormattedText("%.1f", f.value)
    end
end

--- StyleActionButton
-- Glossize DR icons
local function StyleActionButton(f)
    local name = f:GetName()
    local button = _G[name]
    local icon = _G[name .. "Icon"]
    local normalTex = _G[name .. "NormalTexture"]

    normalTex:SetHeight(button:GetHeight())
    normalTex:SetWidth(button:GetWidth())
    normalTex:SetPoint("CENTER", 0, 0)

    button:SetNormalTexture("Interface\\AddOns\\Gladdy\\statusbar\\gloss")

    icon:SetTexCoord(.1, .9, .1, .9)
    icon:SetPoint("TOPLEFT", button, "TOPLEFT", 2, -2)
    icon:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", -2, 2)

    normalTex:SetVertexColor(1, 1, 1, 1)
end

--- Gladdy:ToggleFrame
-- Toggle the frame
function Gladdy:ToggleFrame(i)
    self:ClearAllUnits()
    
    if (self.frame and self.frame:IsShown() and i == self.currentBracket) then
        self:HideFrame()
    else
        self.currentBracket = i
        
        if (not self.frame) then
            self:CreateFrame()
        end
        
        self:UnregisterAllEvents()
        self:RegisterEvent("PLAYER_TARGET_CHANGED")
        self:RegisterEvent("PLAYER_FOCUS_CHANGED")
        self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
        self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
        self:RegisterEvent("UNIT_AURA")

        self.frame:Show()
        self.test = true
        self:Test()

        self:UpdateFrame()
        self:UpdateBindings()
    end
end

--- Gladdy:HideFrame
-- Hide frame and unregister all events/handlers/timers/etc ..
function Gladdy:HideFrame()
    if (self.frame) then
        self.frame:Hide()
    end
    
    self:UnregisterAllEvents()
    self:CancelAllTimers()
    self:UnhookAll()
    self:UnregisterAllComm()

    self.currentBracket = nil
    self.currentUnit = 1
    self.childFrames = 0
    self.clickCount = 0
    self.test = false

    self:RegisterEvent("ZONE_CHANGED_NEW_AREA", "OnZoneChanged")
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "OnZoneChanged")
end

--- Gladdy:CreateFrame
-- Create main frame
function Gladdy:CreateFrame()
    local currentBracket = self.currentBracket or 5
    
    local classIconSize = self.db.barHeight
    local height = (self.db.barHeight * currentBracket) + ((currentBracket - 1) * self.db.barBottomMargin) + (self.db.padding * 2) + 5
    local width = self.db.barWidth + (self.db.padding * 2) + 5 + classIconSize
    
    if (self.db.powerBar) then
        height = height + (currentBracket * self.db.manaBarHeight)
    end
    
    if (self.db.castBar) then
        height = height + (currentBracket * self.db.castBarHeight)
    end
    
    if (self.db.trinketStatus) then
        width = width + classIconSize
    end
    
    self.frame = CreateFrame("Frame", "GladdyFrame", UIParent)
    self.frame:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", tile = true, tileSize = 16 })
    self.frame:SetBackdropColor(self.db.frameColor.r, self.db.frameColor.g, self.db.frameColor.b, self.db.frameColor.a)
    self.frame:SetScale(self.db.frameScale)
    self.frame:SetWidth(width)
    self.frame:SetHeight(height)
    self.frame:SetClampedToScreen(true)
    self.frame:EnableMouse(true)
    self.frame:SetMovable(true)
    self.frame:RegisterForDrag("LeftButton")
    
    self.frame:ClearAllPoints()
    
    if (self.db.x == 0 and self.db.y == 0) then
        self.frame:SetPoint("TOP", UIParent, "TOP", 0, -15)
    else
        local scale = self.frame:GetEffectiveScale()
        self.frame:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", self.db.x / scale, self.db.y / scale)
    end
    
    self.frame:SetScript("OnDragStart", function(f)
        if (not InCombatLockdown() and not self.db.locked) then
            f:StartMoving()
        end
    end)
    
    self.frame:SetScript("OnDragStop", function(f)
        if (not InCombatLockdown()) then
            f:StopMovingOrSizing()
            
            local scale = self.frame:GetEffectiveScale()
            self.db.x = self.frame:GetLeft() * scale
            self.db.y = self.frame:GetTop() * scale
        end
    end)
    
    self.anchor = CreateFrame("Button", nil, self.frame)
    self.anchor:SetWidth(width)
    self.anchor:SetHeight(15)
    self.anchor:SetBackdrop({ bgFile = "Interface\\Tooltips\\UI-Tooltip-Background", tile = true, tileSize = 16 })
    self.anchor:SetBackdropColor(0, 0, 0, 1)
    self.anchor:EnableMouse(true)
    self.anchor:RegisterForClicks("RightButtonUp")
    self.anchor:RegisterForDrag("LeftButton")
    self.anchor:SetClampedToScreen(true)
    
    self.anchor:SetScript("OnDragStart", function(f)
        if (not InCombatLockdown() and not self.db.locked) then
            self.frame:StartMoving()
        end
    end)
    
    self.anchor:SetScript("OnClick", function(f)
        if (not InCombatLockdown()) then
            self:ShowOptions()
        end
    end)
    
    self.anchor.text = self.anchor:CreateFontString(nil, "ARTWORK", "GameFontHighlightSmall")
    self.anchor.text:SetText(L["Gladdy - drag to move"])
    self.anchor.text:SetPoint("CENTER", self.anchor, "CENTER")
    
    self.anchor.button = CreateFrame("Button", nil, self.anchor, "UIPanelCloseButton")
    self.anchor.button:SetHeight(20)
    self.anchor.button:SetWidth(20)
    self.anchor.button:SetPoint("RIGHT", self.anchor, "RIGHT", 2, 0)
    self.anchor.button:SetScript("OnClick", function(frame, button, down)
        if (not down) then
            self.db.locked = true
            self:UpdateFrame()
        end
    end)

    if (self.db.locked) then
        self.anchor:Hide()
    end

    self.frame:Hide()
end

--- Gladdy:CreateButton
-- Create button
function Gladdy:CreateButton(i)
    local classIconSize = self.db.barHeight
    
    if (not self.frame) then
        self:CreateFrame()
    end
    
    local button = CreateFrame("Frame", "GladdyButtonFrame" .. i, self.frame)
    button:Show()
    button:SetAlpha(0)
    
    local targetBorder = CreateFrame("Frame", nil, button)
    targetBorder:SetBackdrop({ edgeFile = "Interface\\ChatFrame\\ChatFrameBackground", edgeSize = 1 })
    targetBorder:SetFrameStrata("HIGH")
    targetBorder:Hide()

    local focusBorder = CreateFrame("Frame", nil, button)
    focusBorder:SetBackdrop({ edgeFile = "Interface\\ChatFrame\\ChatFrameBackground", edgeSize = 1 })
    focusBorder:SetFrameStrata("LOW")
    focusBorder:Hide()

    local leaderBorder = CreateFrame("Frame", nil, button)
    leaderBorder:SetBackdrop({ edgeFile = "Interface\\ChatFrame\\ChatFrameBackground", edgeSize = 1 })
    leaderBorder:SetFrameStrata("MEDIUM")
    leaderBorder:Hide()
    
    local healthBar = CreateFrame("StatusBar", nil, button)
    healthBar:ClearAllPoints()
    healthBar:SetPoint("TOPLEFT", button, "TOPLEFT", classIconSize, 0)
    healthBar:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT")
    healthBar:SetMinMaxValues(0, 100)

    healthBar.bg = healthBar:CreateTexture(nil, "BACKGROUND")
    healthBar.bg:ClearAllPoints()
    healthBar.bg:SetAllPoints(healthBar)
    healthBar.bg:SetAlpha(.3)

    healthBar.highlight = healthBar:CreateTexture(nil, "OVERLAY")
    healthBar.highlight:SetTexture("Interface\\QuestFrame\\UI-QuestTitleHighlight")
    healthBar.highlight:SetBlendMode("ADD")
    healthBar.highlight:SetAlpha(0.5)
    healthBar.highlight:ClearAllPoints()
    healthBar.highlight:SetAllPoints(healthBar)
    healthBar.highlight:Hide()
    
    local classIcon = button:CreateTexture(nil, "ARTWORK")
    classIcon:ClearAllPoints()
    classIcon:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 0)
    
    local manaBar = CreateFrame("StatusBar", nil, button)
    manaBar:ClearAllPoints()
    manaBar:SetPoint("TOPLEFT", healthBar, "BOTTOMLEFT", 0, -1)
    manaBar:SetMinMaxValues(0, 100)
    manaBar:Hide()

    manaBar.bg = manaBar:CreateTexture(nil, "BACKGROUND")
    manaBar.bg:ClearAllPoints()
    manaBar.bg:SetAllPoints(manaBar)
    manaBar.bg:SetAlpha(.3)
    
    if (self.db.powerBar) then
        manaBar:Show()
    end
    
    local castBar = CreateFrame("StatusBar", nil, button)
    castBar:SetMinMaxValues(0, 100)
    castBar:SetValue(0)
    castBar:SetScript("OnUpdate", CastUpdate)
    castBar:Hide()

    castBar.bg = castBar:CreateTexture(nil, "BACKGROUND")
    castBar.bg:ClearAllPoints()
    castBar.bg:SetAllPoints(castBar)

    castBar.icon = castBar:CreateTexture(nil)
    castBar.icon:ClearAllPoints()
    castBar.icon:SetPoint("RIGHT", castBar, "LEFT")
    castBar.icon:SetTexCoord(.1, .9, .1, .9)
    
    if (self.db.castBar) then
        castBar:Show()
    end
    
    local auraFrame = CreateFrame("Frame", nil, button)
    auraFrame:ClearAllPoints()
    auraFrame:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 0)
    auraFrame:SetScript("OnUpdate", AuraUpdate)

    auraFrame.icon = auraFrame:CreateTexture(nil, "ARTWORK")
    auraFrame.icon:SetAllPoints(auraFrame)

    auraFrame.text = auraFrame:CreateFontString(nil, "OVERLAY")
    auraFrame.text:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.auraFontSize)
    auraFrame.text:SetTextColor(self.db.auraFontColor.r, self.db.auraFontColor.g, self.db.auraFontColor.b, self.db.auraFontColor.a)
    auraFrame.text:SetShadowOffset(1, -1)
    auraFrame.text:SetShadowColor(0, 0, 0, 1)
    auraFrame.text:SetJustifyH("CENTER")
    auraFrame.text:SetPoint("CENTER", 0, 0)
    
    local drCooldownFrame = CreateFrame("Frame", nil, button)
    drCooldownFrame:SetFrameStrata("LOW")
    
    for x = 1, 16 do
        local icon = CreateFrame("CheckButton", "Gladdy" .. i .. "DRCooldownFrame" .. x, drCooldownFrame, "ActionButtonTemplate")
        icon:EnableMouse(false)
        icon:SetFrameStrata("BACKGROUND")
        icon.texture = _G[icon:GetName() .. "Icon"]
        icon.cooldown = _G[icon:GetName() .. "Cooldown"]
        icon.cooldown:SetReverse(false)
        icon.cooldown:SetFrameStrata("BACKGROUND")
        icon.text = drCooldownFrame:CreateFontString(nil, "OVERLAY")
        icon.text:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.drFontSize)
        icon.text:SetDrawLayer("OVERLAY")
        icon.text:SetJustifyH("CENTER")
        icon.text:SetPoint("CENTER", icon)
        icon.timeLeftText = drCooldownFrame:CreateFontString(nil, "OVERLAY")
        icon.timeLeftText:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.drFontSize)
        icon.timeLeftText:SetDrawLayer("OVERLAY")
        icon.timeLeftText:SetJustifyH("CENTER")
        icon.timeLeftText:SetPoint("BOTTOM", icon)
        icon.dr = nil
        drCooldownFrame["icon" .. x] = icon
    end
    
    local text = healthBar:CreateFontString(nil, "OVERLAY")
    text:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.healthFontSize)
    text:SetTextColor(self.db.healthFontColor.r, self.db.healthFontColor.g, self.db.healthFontColor.b, self.db.healthFontColor.a)
    text:SetShadowOffset(1, -1)
    text:SetShadowColor(0, 0, 0, 1)
    text:SetJustifyH("LEFT")
    text:SetPoint("LEFT", 5, 0)
    
    local bigTrinket = button:CreateTexture(nil, "ARTWORK")
    local cooldownFrame = CreateFrame("Cooldown", nil, button, "CooldownFrameTemplate")

    local healthText = healthBar:CreateFontString(nil, "LOW")
    healthText:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.healthFontSize)
    healthText:SetTextColor(self.db.healthFontColor.r, self.db.healthFontColor.g, self.db.healthFontColor.b, self.db.healthFontColor.a)
    healthText:SetShadowOffset(1, -1)
    healthText:SetShadowColor(0, 0, 0, 1)
    healthText:SetJustifyH("RIGHT")
    healthText:SetPoint("RIGHT", -5, 0)

    local manaText = manaBar:CreateFontString(nil, "LOW")
    manaText:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.manaFontSize)
    manaText:SetTextColor(self.db.manaFontColor.r, self.db.manaFontColor.g, self.db.manaFontColor.b, self.db.manaFontColor.a)
    manaText:SetShadowOffset(1, -1)
    manaText:SetShadowColor(0, 0, 0, 1)
    manaText:SetJustifyH("CENTER")
    manaText:SetPoint("RIGHT", -5, 0)

    local raceText = manaBar:CreateFontString(nil, "LOW")
    raceText:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.manaFontSize)
    raceText:SetTextColor(self.db.manaFontColor.r, self.db.manaFontColor.g, self.db.manaFontColor.b, self.db.manaFontColor.a)
    raceText:SetShadowOffset(1, -1)
    raceText:SetShadowColor(0, 0, 0, 1)
    raceText:SetJustifyH("CENTER")
    raceText:SetPoint("LEFT", 5, 0)

    local castBarTextSpell = castBar:CreateFontString(nil, "LOW")
    castBarTextSpell:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.castBarFontSize)
    castBarTextSpell:SetTextColor(self.db.castBarFontColor.r, self.db.castBarFontColor.g, self.db.castBarFontColor.b, self.db.castBarFontColor.a)
    castBarTextSpell:SetShadowOffset(1, -1)
    castBarTextSpell:SetShadowColor(0, 0, 0, 1)
    castBarTextSpell:SetJustifyH("CENTER")
    castBarTextSpell:SetPoint("LEFT", 5, 1)

    local castBarTextTime = castBar:CreateFontString(nil, "LOW")
    castBarTextTime:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.castBarFontSize)
    castBarTextTime:SetTextColor(self.db.castBarFontColor.r, self.db.castBarFontColor.g, self.db.castBarFontColor.b, self.db.castBarFontColor.a)
    castBarTextTime:SetShadowOffset(1, -1)
    castBarTextTime:SetShadowColor(0, 0, 0, 1)
    castBarTextTime:SetJustifyH("CENTER")
    castBarTextTime:SetPoint("RIGHT", -5, 1)

    local secure = CreateFrame("Button", "GladdyButton" .. i, button, "SecureActionButtonTemplate")
    secure:RegisterForClicks("AnyUp")
    secure:SetAttribute("*type*", "macro")

    button.mana = manaBar
    button.health = healthBar
    button.castBar = castBar
    button.castBar.timeText = castBarTextTime
    button.castBar.spellText = castBarTextSpell
    button.manaText = manaText
    button.healthText = healthText
    button.text = text
    button.cooldownFrame = cooldownFrame
    button.bigTrinket = bigTrinket
    button.raceText = raceText
    button.classIcon = classIcon
    button.auraFrame = auraFrame
    button.drCooldownFrame = drCooldownFrame
    button.highlight = healthBar.highlight
    button.targetBorder = targetBorder
    button.focusBorder = focusBorder
    button.leaderBorder = leaderBorder
    button.secure = secure
    button.id = i
    button.unit = "arena" .. i
    
    button.name = nil
    button.GUID = nil
    button.spec = ""
    button.spells = {}
    button.dr = {}
    button.dispel = {}
    button.damaged = nil
    button.click = false
    button.enemyAnnounced = false
    button.ns = false -- Nature's Swiftness
    button.nf = false -- Nightfall 
    button.pom = false -- Presence of Mind
    button.lastAura = nil
    button.lowHealth = false
    button._ch = 0
    button._mh = 0
    button._cm = 0
    button._mm = 0
    button._pt = 0

    return button
end

--- Gladdy:UpdateFrame
-- Update whole frame
function Gladdy:UpdateFrame()
    if (not self.frame or not self.currentBracket) then
        return
    end
    
    local currentBracket = self.currentBracket
    
    local classIconSize = self.db.barHeight
    local margin = self.db.barBottomMargin
    local height = (self.db.barHeight * currentBracket) + ((currentBracket - 1) * margin) + (self.db.padding * 2) + 5
    local width = self.db.barWidth + (self.db.padding * 2) + 5 + classIconSize
    local extraWidth = classIconSize
    local extraHeight = 0
    local extraSelectedWidth = classIconSize
    local selectedHeight = self.db.barHeight + 6
    local offset = 0
    
    if (self.db.powerBar) then
        classIconSize = classIconSize + self.db.manaBarHeight
        margin = margin + self.db.manaBarHeight
        height = height + (currentBracket * self.db.manaBarHeight)
        extraHeight = extraHeight + self.db.manaBarHeight
        selectedHeight = selectedHeight + self.db.manaBarHeight
    end
    
    if (self.db.castBar) then
        margin = margin + self.db.castBarHeight
        height = height + (currentBracket * self.db.castBarHeight)
        extraHeight = extraHeight + self.db.castBarHeight
        selectedHeight = selectedHeight + self.db.castBarHeight + 3
    end
    
    if (self.db.trinketStatus) then
        width = width + classIconSize
        extraSelectedWidth = extraSelectedWidth + classIconSize
        offset = offset + (classIconSize / 2)
    end
    
    self.frame:ClearAllPoints()
    
    if (self.db.x == 0 and self.db.y == 0) then
        self.frame:SetPoint("TOP", UIParent, "TOP", 0, -15)
    else
        local scale = self.frame:GetEffectiveScale()

        if (not self.db.growUp) then
            self.frame:SetPoint("TOPLEFT", UIParent, "BOTTOMLEFT", self.db.x / scale, self.db.y / scale)
        else
            self.frame:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", self.db.x / scale, self.db.y / scale)
        end
    end
    
    self.frame:SetScale(self.db.frameScale)
    self.frame:SetWidth(width)
    self.frame:SetHeight(height)
    self.frame:SetBackdropColor(self.db.frameColor.r, self.db.frameColor.g, self.db.frameColor.b, self.db.frameColor.a)

    if (self.db.locked) then
        self.anchor:Hide()
    else
        self.anchor:Show()
    end
    
    self.anchor:SetWidth(width)
    self.anchor:ClearAllPoints()

    self.anchor:SetPoint("TOPLEFT", self.frame, "TOPLEFT", 0, 15)
    
    self.anchor:SetScript("OnDragStop", function(f)
        if (not InCombatLockdown()) then
            self.frame:StopMovingOrSizing()
            local scale = self.frame:GetEffectiveScale()
            self.db.x = self.frame:GetLeft() * scale
            self.db.y = self.db.growUp and self.frame:GetBottom() * scale or self.frame:GetTop() * scale
        end
    end)
    
    for i = 1, currentBracket do
        local button = self.buttons["arena" .. i]
        
        button:SetHeight(self.db.barHeight)
        button:SetWidth(self.db.barWidth + extraWidth)

        button.secure:SetHeight(self.db.barHeight + extraHeight)
        button.secure:SetWidth(self.db.barWidth + extraWidth)
        
        button:ClearAllPoints()
        button.secure:ClearAllPoints()
        
        if (not self.db.growUp) then
            if (i == 1) then
                button:SetPoint("TOPLEFT", self.frame, "TOPLEFT", self.db.padding + 2, -self.db.padding)
                button.secure:SetPoint("TOPLEFT", self.frame, "TOPLEFT", self.db.padding + 2, -self.db.padding)
            else
                button:SetPoint("TOPLEFT", self.buttons["arena" .. (i - 1)], "BOTTOMLEFT", 0, -margin)
                button.secure:SetPoint("TOPLEFT", self.buttons["arena" .. (i - 1)], "BOTTOMLEFT", 0, -margin)
            end
        else
            if (i == 1) then
                button:SetPoint("BOTTOMLEFT", self.frame, "BOTTOMLEFT", self.db.padding + 2, self.db.padding + extraHeight)
                button.secure:SetPoint("BOTTOMLEFT", self.frame, "BOTTOMLEFT", self.db.padding + 2, self.db.padding)
            else
                button:SetPoint("BOTTOMLEFT", self.buttons["arena" .. (i - 1)], "TOPLEFT", 0, margin)
                button.secure:SetPoint("BOTTOMLEFT", self.buttons["arena" .. (i - 1)], "TOPLEFT", 0, margin - extraHeight)
            end
        end
        
        button.targetBorder:SetHeight(selectedHeight)
        button.targetBorder:SetWidth(self.db.barWidth + extraSelectedWidth + 9)
        button.targetBorder:ClearAllPoints()
        button.targetBorder:SetPoint("TOP", button, "TOP", offset, 3)
        button.targetBorder:SetBackdropBorderColor(self.db.targetBorderColor.r, self.db.targetBorderColor.g, self.db.targetBorderColor.b, self.db.targetBorderColor.a)

        button.focusBorder:SetHeight(selectedHeight)
        button.focusBorder:SetWidth(self.db.barWidth + extraSelectedWidth + 9)
        button.focusBorder:ClearAllPoints()
        button.focusBorder:SetPoint("TOP", button, "TOP", offset, 3)
        button.focusBorder:SetBackdropBorderColor(self.db.focusBorderColor.r, self.db.focusBorderColor.g, self.db.focusBorderColor.b, self.db.focusBorderColor.a)

        button.leaderBorder:SetHeight(selectedHeight)
        button.leaderBorder:SetWidth(self.db.barWidth + extraSelectedWidth + 9)
        button.leaderBorder:ClearAllPoints()
        button.leaderBorder:SetPoint("TOP", button, "TOP", offset, 3)
        button.leaderBorder:SetBackdropBorderColor(self.db.leaderBorderColor.r, self.db.leaderBorderColor.g, self.db.leaderBorderColor.b, self.db.leaderBorderColor.a)
        
        button.health:ClearAllPoints()
        button.health:SetPoint("TOPLEFT", button, "TOPLEFT", classIconSize, 0)
        button.health:SetPoint("BOTTOMRIGHT", button, "BOTTOMRIGHT", 0, 0)
        button.health:SetStatusBarTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))
        button.health.bg:SetTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))

        button.mana:ClearAllPoints()
        button.mana:SetHeight(self.db.manaBarHeight)
        button.mana:SetWidth(button.health:GetWidth())
        button.mana:SetPoint("TOPLEFT", button.health, "BOTTOMLEFT", 0, -1)
        button.mana:SetStatusBarTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))
        button.mana.bg:SetTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))

        if (not self.db.powerBar) then
            button.mana:Hide()
        else
            button.mana:Show()
        end
        
        button.castBar:ClearAllPoints()
        local castBarX = self.db.castBarHeight - classIconSize
        local parent = self.db.powerBar and button.mana or button.health
        button.castBar:SetPoint("TOPLEFT", parent, "BOTTOMLEFT", castBarX, 0)
        button.castBar:SetHeight(self.db.castBarHeight)
        button.castBar:SetWidth(button.health:GetWidth() + classIconSize - self.db.castBarHeight)

        button.castBar:SetStatusBarTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))
        button.castBar:SetStatusBarColor(self.db.castBarColor.r, self.db.castBarColor.g, self.db.castBarColor.b, self.db.castBarColor.a)
        button.castBar.bg:SetTexture(LSM:Fetch(LSM.MediaType.STATUSBAR, self.db.barTexture))
        button.castBar.bg:SetVertexColor(self.db.castBarBgColor.r, self.db.castBarBgColor.g, self.db.castBarBgColor.b, self.db.castBarBgColor.a)

        button.castBar.icon:SetHeight(self.db.castBarHeight)
        button.castBar.icon:SetWidth(self.db.castBarHeight)
        button.castBar.bg:ClearAllPoints()
        button.castBar.bg:SetPoint("RIGHT", button.castBar, "RIGHT")
        button.castBar.bg:SetWidth(button.castBar:GetWidth() + self.db.castBarHeight)
        button.castBar.bg:SetHeight(button.castBar:GetHeight())

        if (not self.db.castBar) then
            button.castBar:Hide()
        else
            button.castBar:Show()
        end
        
        button.text:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.healthFont), self.db.healthFontSize)
        button.healthText:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.healthFont), self.db.healthFontSize)
        button.manaText:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.manaFont), self.db.manaFontSize)
        button.raceText:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.manaFont), self.db.manaFontSize)
        button.castBar.spellText:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.castBarFont), self.db.castBarFontSize)
        button.castBar.timeText:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.castBarFont), self.db.castBarFontSize)
        
        if (button.class) then
            button.health:SetStatusBarColor(RAID_CLASS_COLORS[button.class].r, RAID_CLASS_COLORS[button.class].g, RAID_CLASS_COLORS[button.class].b, 1)
        else
            button.health:SetStatusBarColor(1, 1, 1, 1)
        end
        
        if (button._pt == 0 and not self.db.manaDefault) then
            button.mana:SetStatusBarColor(self.db.manaColor.r, self.db.manaColor.g, self.db.manaColor.b, self.db.manaColor.a)
        elseif (button._pt == 1 and not self.db.rageDefault) then
            button.mana:SetStatusBarColor(self.db.rageColor.r, self.db.rageColor.g, self.db.rageColor.b, self.db.rageColor.a)
        elseif (button._pt == 3 and not self.db.energyDefault) then
            button.mana:SetStatusBarColor(self.db.energyColor.r, self.db.energyColor.g, self.db.energyColor.b, self.db.energyColor.a)
        else
            button.mana:SetStatusBarColor(POWER_COLOR[button._pt].r, POWER_COLOR[button._pt].g, POWER_COLOR[button._pt].b, POWER_COLOR[button._pt].a)
        end
        
        button.classIcon:ClearAllPoints()
        button.classIcon:SetWidth(classIconSize)
        button.classIcon:SetHeight(classIconSize + 1)
        button.classIcon:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 0)
        button.classIcon:Show()

        button.auraFrame:ClearAllPoints()
        button.auraFrame:SetWidth(classIconSize)
        button.auraFrame:SetHeight(classIconSize + 1)
        button.auraFrame:SetPoint("TOPLEFT", button, "TOPLEFT", -2, 0)
        
        if (not self.db.specText and not self.db.raceText) then
            button.raceText:Hide()
        else
            button.raceText:Show()
        end

        if (not self.db.manaText) then
            button.manaText:Hide()
        else
            button.manaText:Show()
        end

        button.bigTrinket:SetWidth(classIconSize)
        button.bigTrinket:SetHeight((classIconSize + 1))
        button.bigTrinket:SetTexture("Interface\\Icons\\INV_Jewelry_TrinketPVP_02")
        button.bigTrinket:ClearAllPoints()
        button.bigTrinket:SetPoint("TOPRIGHT", button, "TOPRIGHT", classIconSize + 1, 0)

        if (not self.db.trinketStatus) then
            button.bigTrinket:Hide()
        else
            button.bigTrinket:Show()
        end

        if (self.db.trinketStatus) then
            button.cooldownFrame:ClearAllPoints()
            button.cooldownFrame:SetWidth(classIconSize)
            button.cooldownFrame:SetHeight(classIconSize + 1)
            button.cooldownFrame:SetAllPoints(button.bigTrinket)
            button.cooldownFrame:Show()
        else
            button.cooldownFrame:Hide()
        end
        
        if (self.db.drCooldown) then
            local anchor = self.db.drCooldownAnchor == "CENTER" and "" or self.db.drCooldownAnchor

            button.drCooldownFrame:ClearAllPoints()

            if (self.db.drCooldownPos == "RIGHT") then
                if (self.db.trinketStatus) then
                    button.drCooldownFrame:SetPoint("TOPLEFT", button.bigTrinket, "TOPRIGHT", 5, -1)
                else
                    button.drCooldownFrame:SetPoint("TOPLEFT", button, "TOPRIGHT", 5, -1)
                end
            else
                button.drCooldownFrame:SetPoint("TOPRIGHT", button, "TOPLEFT", -5, -1)
            end

            button.drCooldownFrame:SetHeight(self.db.barHeight + extraHeight)
            button.drCooldownFrame:SetWidth(self.db.barHeight + extraHeight)

            for i = 1, 16 do
                local icon = button.drCooldownFrame["icon" .. i]

                if (self.db.drIconAdjust) then
                    icon:SetHeight(button.drCooldownFrame:GetHeight())
                    icon:SetWidth(button.drCooldownFrame:GetWidth())
                else
                    icon:SetHeight(self.db.drIconSize)
                    icon:SetWidth(self.db.drIconSize)
                end

                icon.cooldown.noCooldownCount = true

                icon.text:SetFont(LSM:Fetch(LSM.MediaType.FONT, self.db.drFont), self.db.drFontSize, "OUTLINE")
                icon.text:SetTextColor(self.db.drFontColor.r, self.db.drFontColor.g, self.db.drFontColor.b, self.db.drFontColor.a)
                icon.timeLeftText:SetFont(LSM:Fetch(LSM.MediaType.FONT), self.db.drFontSize - 3, "OUTLINE")
                icon.timeLeftText:SetTextColor(self.db.drFontColor.r, self.db.drFontColor.g, self.db.drFontColor.b, self.db.drFontColor.a)
                icon:ClearAllPoints()

                if (self.db.drCooldownPos == "RIGHT") then
                    if (i == 1) then
                        icon:SetPoint(anchor .. "LEFT", button.drCooldownFrame)
                    else
                        icon:SetPoint("LEFT", button.drCooldownFrame["icon" .. (i - 1)], "RIGHT", 05, 0)
                    end
                else
                    if (i == 1) then
                        icon:SetPoint(anchor .. "RIGHT", button.drCooldownFrame)
                    else
                        icon:SetPoint("RIGHT", button.drCooldownFrame["icon" .. (i - 1)], "LEFT", 0, 0)
                    end
                end

                if (icon.active) then
                    icon.active = false
                    icon.cooldown:SetCooldown(GetTime(), 0)
                    icon:SetScript("OnUpdate", nil)
                end

                icon.dr = nil
                icon.active = nil
                icon:SetAlpha(1)
                icon.text:SetText("")
                icon.texture:SetTexture(select(3, GetSpellInfo(118)))
                StyleActionButton(icon)

                if (not self.test or i > 5) then
                    icon:Hide()
                else
                    if (self.db.drText) then
                        icon.text:SetText("1/2")
                        icon.timeLeftText:SetText("15")
                    end

                    icon.active = true
                    icon:Show()
                end
            end
            button.drCooldownFrame:Show()
        else
            button.drCooldownFrame:Hide()
        end
        
        if (self.test) then
            local ch, mh = button._ch, button._mh
            local cm, mm, pt = button._cm, button._mm, button._pt
            
            self:HandleUpdateHealth(button.unit, ch, mh)
            self:HandleUpdatePower(button.unit, cm, mm, pt)
        end
    end
    
    self:PLAYER_TARGET_CHANGED()
    self:PLAYER_FOCUS_CHANGED()
end
