-- Create addon
LibStub("AceAddon-3.0"):NewAddon("Gladdy", "AceEvent-3.0", "AceConsole-3.0", "AceTimer-3.0", "AceHook-3.0", "AceComm-3.0")

-- Get core objects
local Gladdy = LibStub("AceAddon-3.0"):GetAddon("Gladdy")
local L = LibStub("AceLocale-3.0"):GetLocale("Gladdy")

-- Lua calls
local pairs = pairs
local setmetatable = setmetatable
local rawset = rawset
local select = select
local tonumber = tonumber
local floor = math.floor

-- Blizzard API
local GetSpellInfo = GetSpellInfo
local IsAddOnLoaded = IsAddOnLoaded
local IsLoggedIn = IsLoggedIn
local IsInInstance = IsInInstance
local GetBattlefieldStatus = GetBattlefieldStatus
local MAX_BATTLEFIELD_QUEUES = MAX_BATTLEFIELD_QUEUES
local UnitBuff = UnitBuff
local UnitDebuff = UnitDebuff
local RAID_CLASS_COLORS = RAID_CLASS_COLORS
local GetTime = GetTime
local GetRealNumPartyMembers = GetRealNumPartyMembers
local GetRealNumRaidMembers = GetRealNumRaidMembers
local SendChatMessage = SendChatMessage
local RaidNotice_AddMessage = RaidNotice_AddMessage
local RaidBossEmoteFrame = RaidBossEmoteFrame
local CombatText_AddMessage = CombatText_AddMessage
local COMBAT_TEXT_SCROLL_FUNCTION = COMBAT_TEXT_SCROLL_FUNCTION
local COMBATLOG_OBJECT_REACTION_HOSTILE  = COMBATLOG_OBJECT_REACTION_HOSTILE
local UnitIsDeadOrGhost = UnitIsDeadOrGhost
local UnitHealth = UnitHealth
local UnitHealthMax = UnitHealthMax
local UnitMana = UnitMana
local UnitManaMax = UnitManaMax
local UnitPowerType = UnitPowerType
local UnitCastingInfo= UnitCastingInfo
local UnitChannelInfo = UnitChannelInfo
local UnitIsPartyLeader = UnitIsPartyLeader
local UnitName = UnitName
local UnitGUID = UnitGUID
local GetBindingKey = GetBindingKey
local ClearOverrideBindings = ClearOverrideBindings
local SetOverrideBindingClick = SetOverrideBindingClick
local UnitExists = UnitExists
local UnitIsPlayer = UnitIsPlayer
local UnitCanAttack = UnitCanAttack
local UnitIsCharmed = UnitIsCharmed
local UnitClass = UnitClass
local UnitRace = UnitRace
local UNKNOWNOBJECT = UNKNOWNOBJECT
local unpack = unpack
local CLASS_BUTTONS = CLASS_BUTTONS
local tostring = tostring
local CombatLogClearEntries = CombatLogClearEntries
local CreateFrame = CreateFrame
local CooldownFrame_SetTimer = CooldownFrame_SetTimer
local WorldFrame = WorldFrame

-- Other globals
local ClickCastFrames = ClickCastFrames
local MikSBT = MikSBT
local SCT = SCT
local Parrot = Parrot

-- Local constants
local NS = GetSpellInfo(16188)
local NF = GetSpellInfo(18095)
local POM = GetSpellInfo(12043)
local CORRUPTION = GetSpellInfo(172)
local SHADOWBOLT = GetSpellInfo(27209)
local DR = 15
local DRINK = GetSpellInfo(46755)
local RES = {
    [GetSpellInfo(20770)] = true,
    [GetSpellInfo(20773)] = true,
    [GetSpellInfo(20777)] = true,
}
local POWER_COLOR = {
    [0] = { r = .18, g = .44, b = .75, a = 1 },
    [1] = { r = 1, g = 0, b = 0, a = 1 },
    [3] = { r = 1, g = 1, b = 0, a = 1 },
}
local DR_TIME = { "1/2", "1/4", "0" }
local TRINKET_ICON = "Interface\\Icons\\INV_Jewelry_TrinketPVP_02"

-- Update bindings menu
_G["BINDING_HEADER_GLADDY"] = L["Gladdy"]
_G["BINDING_NAME_GLADDYBUTTON1_LEFT"] = L["Left Click Enemy 1"]
_G["BINDING_NAME_GLADDYBUTTON2_LEFT"] = L["Left Click Enemy 2"]
_G["BINDING_NAME_GLADDYBUTTON3_LEFT"] = L["Left Click Enemy 3"]
_G["BINDING_NAME_GLADDYBUTTON4_LEFT"] = L["Left Click Enemy 4"]
_G["BINDING_NAME_GLADDYBUTTON5_LEFT"] = L["Left Click Enemy 5"]
_G["BINDING_NAME_GLADDYBUTTON1_RIGHT"] = L["Right Click Enemy 1"]
_G["BINDING_NAME_GLADDYBUTTON2_RIGHT"] = L["Right Click Enemy 2"]
_G["BINDING_NAME_GLADDYBUTTON3_RIGHT"] = L["Right Click Enemy 3"]
_G["BINDING_NAME_GLADDYBUTTON4_RIGHT"] = L["Right Click Enemy 4"]
_G["BINDING_NAME_GLADDYBUTTON5_RIGHT"] = L["Right Click Enemy 5"]

--- Gladdy:OnInitialize
-- Initialize db, variables, options, etc ...
function Gladdy:OnInitialize()
    -- Initialize DB
    self.dbi = LibStub("AceDB-3.0"):New("Gladdy2DB", self.defaults)
    self.dbi.RegisterCallback(self, "OnProfileChanged", "OnProfileChanged")
    self.dbi.RegisterCallback(self, "OnProfileCopied", "OnProfileChanged")
    self.dbi.RegisterCallback(self, "OnProfileReset", "OnProfileChanged")
    
    -- Create shortcut for short DB manipulations
    self.db = self:GetDB()
    
    -- Initialize properties
    self.buttons = {}
    self.throttle = {}
    self.arenaGUID = {}
    self.currentUnit = 1
    self.currentBracket = nil
    self.lastInstance = nil
    self.childFrames = 0
    self.clickCount = 0
    self.refreshTimer = nil
    
    -- Test environment
    self.test = false
    self.testing = setmetatable({
        ["arena2"] = { ch = 12700, mh = 13400, cm = 12100, mm = 12200, pt = 0, cL = L["Warlock"], c = "WARLOCK", rL = L["Gnome"], spec = L["Demonology"] },
        ["arena3"] = { ch = 10000, mh = 10650, cm = 9999, mm = 10210, pt = 0, cL = L["Mage"], c = "MAGE", rL = L["Undead"], spec = L["Frost"] },
        ["arena4"] = { ch = 11230, mh = 11250, cm = 60, mm = 110, pt = 3, cL = L["Rogue"], c = "ROGUE", rL = L["Human"], spec = L["Subtlety"] },
        ["arena5"] = { ch = 14200, mh = 15600, cm = 95, mm = 100, pt = 1, cL = L["Warrior"], c = "WARRIOR", rL = L["Tauren"], spec = L["Arms"] },
    }, {
        __index = function(t)
            return t["arena2"]
        end
    })
    
    -- Initialize options
    self:SetupOptions()
end

--- Gladdy:OnProfileChanged
-- Callback for AceDB on profile change
function Gladdy:OnProfileChanged()
    -- Refresh DB
    self.db = self:GetDB()
    
    -- Display test frame
    self:HideFrame()
    self:ToggleFrame(5)
end

--- Gladdy:GetDB
-- Get shortcut for DB profile
function Gladdy:GetDB()
    return setmetatable(self.dbi.profile, {
        __newindex = function(t, k, v)
            rawset(self.dbi.profile, k, v)
            rawset(t, k, v)
        end
    })
end

--- Gladdy:OnEnable
-- Enable addon handler
function Gladdy:OnEnable()
    -- Register core events
    self:RegisterEvent("ZONE_CHANGED_NEW_AREA", "OnZoneChanged")
    self:RegisterEvent("PLAYER_ENTERING_WORLD", "OnZoneChanged")

    -- Clique support
    if (IsAddOnLoaded("Clique")) then
        for i = 1, 5 do
            self.buttons["arena" .. i] = self:CreateButton(i)
        end

        ClickCastFrames = ClickCastFrames or {}
        ClickCastFrames[self.buttons.arena1.secure] = true
        ClickCastFrames[self.buttons.arena2.secure] = true
        ClickCastFrames[self.buttons.arena3.secure] = true
        ClickCastFrames[self.buttons.arena4.secure] = true
        ClickCastFrames[self.buttons.arena5.secure] = true
    end
    
    -- First launch?
    if (not self.db.locked and self.db.x == 0 and self.db.y == 0) then
        self:HideFrame()
        self:ToggleFrame(5)
        self:Print(L["Welcome to Gladdy!"])
        self:Print(L["First launch detected, displaying test frame"])
        self:Print(L["Valid slash commands are:"])
        self:Print("/gladdy ui")
        self:Print("/gladdy test1-5")
        self:Print("/gladdy hide")
        self:Print(L["If it not first launch, then move or lock frame"])
    end
end 

--- Gladdy:OnZoneChanged
-- Fired when player enters world
function Gladdy:OnZoneChanged()
    self:ClearAllUnits()
    self:ZoneCheck()
end

--- Gladdy:ClearAllUnits
-- Recursively clear all buttons' data
function Gladdy:ClearAllUnits()
    for unit, button in pairs(self.buttons) do
        -- Reset button props
        button.name = nil
        button.GUID = nil
        button.spec = ""
        button.spells = {}
        button.dr = {}
        button.dispel = {}
        button.damaged = nil
        button.click = false
        button.enemyAnnounced = false
        button.ns = false -- Nature's Swiftness
        button.nf = false -- Nightfall 
        button.pom = false -- Presence of Mind
        button.lastAura = nil
        button.lowHealth = false
        button._ch = 0
        button._mh = 0
        button._cm = 0
        button._mm = 0
        button._pt = 0
        
        -- Trinket
        CooldownFrame_SetTimer(button.cooldownFrame, 1, 1, 1)
        
        if (button.trinketFrame) then
            button.trinketFrame:SetScript("OnUpdate", nil)
        end
        
        -- Reset borders
        button.targetBorder:Hide()
        button.focusBorder:Hide()
        button.leaderBorder:Hide()
        
        -- Reset attributes
        for k, v in pairs(self.db.attributes) do
            local attr = v.modifier .. "macrotext" .. v.button
            button.secure:SetAttribute(attr, "")
        end
        
        -- Clear DR icons
        for i = 1, 16 do
            local DrIcon = button.drCooldownFrame["icon" .. i]
            DrIcon:SetAlpha(0)

            if (DrIcon.text:GetText()) then
                DrIcon.text:SetText("")
            end

            if (DrIcon.timeLeftText:GetText()) then
                DrIcon.timeLeftText:SetText("")
            end
        end
        
        -- Hide button
        button:SetAlpha(0)
    end
    
    -- Reset common props
    self.throttle = {}
    self.arenaGUID = {}
    self.currentUnit = 1
    self.currentBracket = nil
    self.childFrames = 0
    self.clickCount = 0
    self.refreshTimer = nil
end

--- Gladdy:ZoneCheck
-- Detect current zone, fire required handlers
function Gladdy:ZoneCheck()
    local zone = select(2, IsInInstance())
    
    if (zone == "arena") then
        self:JoinedArena()
    elseif (zone ~= "arena" and self.lastInstance == "arena") then
        self:HideFrame()
    end
    
    self.lastInstance = zone
end

--- Gladdy:JoinedArena
-- Event fired when player enters arena
function Gladdy:JoinedArena()
    -- Reset env
    self.test = false
    
    -- Detect bracket
    for i = 1, MAX_BATTLEFIELD_QUEUES do
        local status, _, _, _, _, teamSize = GetBattlefieldStatus(i)
        if (status == "active" and teamSize > 0) then
            self.currentBracket = teamSize
            break
        end
    end

    -- Init buttons
    for i = 1, (self.currentBracket or 5) do
        local unit = "arena" .. i
        
        if (not self.buttons[unit]) then
            self.buttons[unit] = self:CreateButton(i)
        end
    end
    
    -- Update frame
    self:UpdateFrame()
    self:UpdateBindings()
    self.frame:Show()

    -- Player events
    self:RegisterEvent("PLAYER_TARGET_CHANGED")
    self:RegisterEvent("PLAYER_FOCUS_CHANGED")
    self:RegisterEvent("UPDATE_MOUSEOVER_UNIT")
    self:RegisterEvent("UNIT_TARGET")
    self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")

    -- Register events
    self:RegisterEvent("UNIT_AURA")
    self:RegisterEvent("UNIT_HEALTH")
    self:RegisterEvent("UNIT_MAXHEALTH", "UNIT_HEALTH")
    self:RegisterEvent("UNIT_POWER")
    self:RegisterEvent("UNIT_MAXPOWER", "UNIT_POWER")
    self:RegisterEvent("UNIT_RAGE", "UNIT_POWER")
    self:RegisterEvent("UNIT_ENERGY", "UNIT_POWER")
    self:RegisterEvent("UNIT_DISPLAYPOWER", "UNIT_POWER")
    self:RegisterEvent("UNIT_SPELLCAST_START")
    self:RegisterEvent("UNIT_SPELLCAST_STOP")
    self:RegisterEvent("UNIT_SPELLCAST_DELAYED")
    self:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
    self:RegisterEvent("UNIT_SPELLCAST_FAILED", "UNIT_SPELLCAST_STOP")
    self:RegisterEvent("UNIT_SPELLCAST_INTERRUPTED", "UNIT_SPELLCAST_STOP")
    self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_START")
    self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_UPDATE", "UNIT_SPELLCAST_DELAYED")
    self:RegisterEvent("UNIT_SPELLCAST_CHANNEL_STOP", "UNIT_SPELLCAST_STOP")

    -- Start timers
    self.refreshTimer = self:ScheduleRepeatingTimer("RefreshFrame", 0.1, self)
    self:ScheduleRepeatingTimer("Sync", 1, self)
    self:ScheduleRepeatingTimer("UpdateNameplates", 1, self)

    -- Comm
    self:RegisterComm("Gladdy2")
    self:RegisterComm("GladdyTrinketUsed")

    -- Cleanup combatlog
    CombatLogClearEntries()
end

--- Gladdy:UpdateBindings
-- Set bindings for buttons
function Gladdy:UpdateBindings()
    for unit, button in pairs(self.buttons) do
        local left = GetBindingKey(("GLADDYBUTTON%d_LEFT"):format(button.id))
        local right = GetBindingKey(("GLADDYBUTTON%d_RIGHT"):format(button.id))

        ClearOverrideBindings(button.secure)

        if (left) then
            SetOverrideBindingClick(button.secure, false, left, button.secure:GetName(), "LeftButton")
        end
        if (right) then
            SetOverrideBindingClick(button.secure, false, right, button.secure:GetName(), "RightButton")
        end
    end
end

--- Gladdy:PLAYER_TARGET_CHANGED
-- Event fired when player changes target
function Gladdy:PLAYER_TARGET_CHANGED()
    local targetGUID = UnitGUID("target")
    
    if (targetGUID ~= nil) then
        self:UpdateUnit("target")
    end
    
    for unit, button in pairs(self.buttons) do
        if (targetGUID ~= nil and button.GUID == targetGUID) then
            if (self.db.highlight) then
                button.highlight:Show()
            end
            
            if (self.db.targetBorder) then
                button.targetBorder:Show()
            end
        else
            button.highlight:Hide()
            button.targetBorder:Hide()
        end
    end
end

--- Gladdy:PLAYER_FOCUS_CHANGED
-- Event fired when player changes focus
function Gladdy:PLAYER_FOCUS_CHANGED()
    local focusGUID = UnitGUID("focus")
    
    if (focusGUID ~= nil) then
        self:UpdateUnit("focus")
    end
    
    for unit, button in pairs(self.buttons) do
        if (focusGUID ~= nil and button.GUID == focusGUID) then
            if (self.db.focusBorder) then
                button.focusBorder:Show()
            end
        else
            button.focusBorder:Hide()
        end
    end
end

--- Gladdy:UPDATE_MOUSEOVER_UNIT
-- Event fired when player hovers unit with cursor
function Gladdy:UPDATE_MOUSEOVER_UNIT()
    self:UpdateUnit("mouseover")
end

--- Gladdy:UNIT_TARGET
-- Update party leader target
function Gladdy:UNIT_TARGET(event, uid)
    if (UnitIsPartyLeader(uid) and UnitName(uid) ~= UnitName("player")) then
        local GUID = UnitGUID(uid .. "target")
        
        for unit, button in pairs(self.buttons) do
            if (GUID ~= nil and button.GUID == GUID) then
                if (self.db.leaderBorder) then
                    button.leaderBorder:Show()
                end
            else
                button.leaderBorder:Hide()
            end
        end
    end
end

--- Gladdy:COMBAT_LOG_EVENT_UNFILTERED
-- CombatLog parser
function Gladdy:COMBAT_LOG_EVENT_UNFILTERED(event, timestamp, eventType, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, spellId, spellName)
    local sourceUnit = self.arenaGUID[sourceGUID]
    local destUnit = self.arenaGUID[destGUID]
    
    if (not sourceUnit and not destUnit) then
        return
    end
    
    local events = {
        ["PARTY_KILL"] = "DEATH",
        ["UNIT_DIED"] = "DEATH",
        ["UNIT_DESTROYED"] = "DEATH",
        ["SWING_DAMAGE"] = "DAMAGE",
        ["RANGE_DAMAGE"] = "DAMAGE",
        ["SPELL_DAMAGE"] = "DAMAGE",
        ["SPELL_PERIODIC_DAMAGE"] = "DAMAGE",
        ["ENVIRONMENTAL_DAMAGE"] = "DAMAGE",
        ["DAMAGE_SHIELD"] = "DAMAGE",
        ["DAMAGE_SPLIT"] = "DAMAGE",
    }
    
    local t = ("%.1f"):format(GetTime())
    
    if (events[eventType] == "DEATH" and destUnit) then
        self:UnitDeath(destUnit)
    elseif (events[eventType] == "DAMAGE" and destUnit) then
        local button = self.buttons[destUnit]
        if (not button) then return end
        
        button.damaged = t
    elseif (eventType == "SPELL_AURA_APPLIED" and destUnit) then
        local button = self.buttons[destUnit]
        local aura = self.auras[spellName]
        if (not button or not aura or not aura.track) then return end
        
        if (aura.dr ~= nil) then
            self:DrGain(destUnit, spellName)
        end
        
        if (aura.priority >= (button.auraFrame.priority or 0)) then
            local dr, factor = aura.dr, 1
                
            if (dr and button.dr[dr] and button.dr[dr] > 0) then
                factor = button.dr[dr]
            end
            
            local icon = select(3, GetSpellInfo(spellId))
            local duration = aura.duration / factor
            
            self:HandleAuraGain(destUnit, spellName, duration)
            self:AuraGain(button.auraFrame, spellName, icon, duration, aura.priority)
            
            button.spells[spellName] = t
        end
    elseif (eventType == "SPELL_AURA_REFRESH" and destUnit) then
        local button = self.buttons[destUnit]
        local aura = self.auras[spellName]
        if (not button or not aura or not aura.track) then return end
        
        if (button.spells[spellName] and t > button.spells[spellName]) then
            if (aura.dr ~= nil) then
                self:DrFades(destUnit, spellName)
                self:DrGain(destUnit, spellName)
            end
            
            if (aura.priority >= (button.auraFrame.priority or 0)) then
                local dr, factor = aura.dr, 1
                
                if (dr and button.dr[dr] and button.dr[dr] > 0) then
                    factor = button.dr[dr]
                end
                
                local icon = select(3, GetSpellInfo(spellId))
                local duration = aura.duration / factor
                
                self:HandleAuraGain(destUnit, spellName, duration)
                self:AuraGain(button.auraFrame, spellName, icon, duration, aura.priority)
                
                button.spells[spellName] = t
            end
        end
    elseif (eventType == "SPELL_AURA_REMOVED" and destUnit) then
        local button = self.buttons[destUnit]
        if (not button) then return end
        
        self:HandleAuraFade(destUnit, spellName)
        
        local aura = self.auras[spellName]
        if (not aura or not aura.track) then return end
        
        button.spells[spellName] = nil
        
        if (button.auraFrame.name == spellName) then
            self:AuraFades(button.auraFrame)
        end
        
        if (aura.dr ~= nil) then
            self:DrFades(destUnit, spellName)
        end
    elseif (eventType == "SPELL_CAST_START" and sourceUnit) then
        local button = self.buttons[sourceUnit]
        if (not button) then return end
        
        if (button.pom or button.ns or (button.nf and spellName == SHADOWBOLT) or spellName == CORRUPTION) then
            return
        end
        
        local auraIcon = select(3, GetSpellInfo(spellId))
        local castTime = select(7, GetSpellInfo(spellId)) / 1000
        
        self:HandleCastStart(sourceUnit, spellName, auraIcon, 0, castTime, nil, "cast")
    elseif (eventType == "SPELL_CAST_SUCCESS" and sourceUnit) then
        local button = self.buttons[sourceUnit]
        if (not button) then return end
        
        local spec = self.spells[spellName]
        
        if (spec ~= nil) then
            self:UpdateSpec(sourceUnit, spec)
        end
        
        if (spellName == NS) then
            button.ns = true
        elseif (spellName == NF) then
            button.nf = true
        elseif (spellName == POM) then
            button.pom = true
        end

        local dispel = self.dispels[spellName]

        if (dispel) then
            button.dispel[spellName] = t
        end
    elseif (eventType == "SPELL_CAST_FAILED") then
        local button = self.buttons[sourceUnit]
        if (not button) then return end
        
        self:HandleCastEnd(button.castBar)
    end
end

--- Gladdy:UNIT_AURA
-- Auras event handler
function Gladdy:UNIT_AURA(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not unit or not button) then return end
    
    local auraFrame = self.buttons[unit].auraFrame
    local index, priority = 1, 0
    local auraName, auraIcon, auraExpTime

    while (true) do
        local name, _, icon, _, _, expTime = UnitBuff(uid, index)
        if (not name) then break end
        
        self:HandleAuraGain(unit, name)
        
        if (self:CheckAura(name, priority, "buff")) then
            auraName = name
            auraIcon = icon
            auraExpTime = expTime or 0
            priority = self.auras[name].priority
        end
        
        index = index + 1
    end
    
    index = 1
    
    while (true) do
        local name, _, icon, _, _, _, expTime = UnitDebuff(uid, index)
        if (not name) then break end
        
        if (self:CheckAura(name, priority, "debuff")) then
            auraName = name
            auraIcon = icon
            auraExpTime = expTime or 0
            priority = self.auras[name].priority
            
            self:HandleAuraGain(unit, name, auraExpTime)
        end
        
        index = index + 1
    end

    if (auraName) then
        if (self.auras[auraName].track == "buff") then
            if (auraName == auraFrame.name) then
                auraExpTime = auraFrame.timeLeft
            else
                auraExpTime = self.auras[auraName].duration
            end
            
            -- Now we have duration for buff
            self:HandleAuraGain(unit, auraName, auraExpTime)
        end
        
        self:AuraGain(auraFrame, auraName, auraIcon, auraExpTime, priority)
    elseif (not auraName and auraFrame.active) then
        self:AuraFades(auraFrame)
    end
end

--- Gladdy:UNIT_HEALTH
-- Health update event
function Gladdy:UNIT_HEALTH(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not unit) then return end
    
    if (not UnitIsDeadOrGhost(uid)) then
        local ch, mh = UnitHealth(uid), UnitHealthMax(uid)
        
        self:HandleUpdateHealth(unit, ch, mh)
    else
        self:UnitDeath(unit)
    end
end

--- Gladdy:UNIT_POWER
-- Power update event
function Gladdy:UNIT_POWER(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not unit) then return end
    
    if (not UnitIsDeadOrGhost(uid)) then
        local cm, mm, pt = UnitMana(uid), UnitManaMax(uid), UnitPowerType(uid)
        self:HandleUpdatePower(unit, cm, mm, pt)
    else
        self:UnitDeath(unit)
    end 
end

--- Gladdy:UNIT_SPELLCAST_START
-- Cast start event
function Gladdy:UNIT_SPELLCAST_START(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not unit or not button) then return end
    
    local spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo(uid)
    local value = (endTime / 1000) - GetTime()
    local maxValue = (endTime - startTime) / 1000
    
    self:HandleCastStart(unit, spell, icon, value, maxValue, rank, "cast")
end

--- Gladdy:UNIT_SPELLCAST_CHANNEL_START
-- Channelling start event
function Gladdy:UNIT_SPELLCAST_CHANNEL_START(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not unit or not button) then return end
    
    local spell, rank, displayName, icon, startTime, endTime = UnitChannelInfo(uid)
    local value = (endTime / 1000) - GetTime()
    local maxValue = (endTime - startTime) / 1000
    
    self:HandleCastStart(unit, spell, icon, value, maxValue, rank, "channel")
end

--- Gladdy:UNIT_SPELLCAST_SUCCEEDED
-- Instantcast event
function Gladdy:UNIT_SPELLCAST_SUCCEEDED(event, uid, spell)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local spec = self.spells[spell]
    if (not spec) then return end
    
    if (self.test and uid == "player") then
        unit = "arena1"
    end
    
    local button = self.buttons[unit]
    if (not button) then return end
    
    if (not button.spec and spec) then
        self:UpdateSpec(unit, spec)
    end
end

--- Gladdy:UNIT_SPELLCAST_STOP
-- Stop cast event
function Gladdy:UNIT_SPELLCAST_STOP(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not button) then return end
    
    self:HandleCastEnd(button.castBar)
end

--- Gladdy:UNIT_SPELLCAST_DELAYED
-- Cast delayed
function Gladdy:UNIT_SPELLCAST_DELAYED(event, uid)
    local unit = self.arenaGUID[UnitGUID(uid)]
    local button = self.buttons[unit]
    if (not button) then return end
    
    local spell, rank, displayName, icon, startTime, endTime
    
    if (event == "UNIT_SPELLCAST_DELAYED") then
        spell, rank, displayName, icon, startTime, endTime = UnitCastingInfo(uid)
    else
        spell, rank, displayName, icon, startTime, endTime = UnitChannelInfo(uid)
    end
    
    local castBar = button.castBar
    castBar.value = GetTime() - (startTime / 1000)
    castBar.maxValue = (endTime - startTime) / 1000
    castBar:SetMinMaxValues(0, castBar.maxValue)
end

--- Gladdy:RefreshFrame
-- Update frame data
function Gladdy:RefreshFrame()
    for unit, button in pairs(self.buttons) do
        if (button.GUID) then
            if (not button.click and not InCombatLockdown()) then
                self:UpdateAttributes(unit)
                button.click = true
                self.clickCount = self.clickCount + 1
            end
            
            if (button.click) then
                button:SetAlpha(1)
                button.text:SetText(button.name)
            else
                button:SetAlpha(0.66)
                button.text:SetText("**" .. button.name .. "**")
            end
        end
    end
    
    -- Updated all attributes, no need to call every 0.1 second
    if (self.clickCount == self.currentBracket and self.refreshTimer) then
        self:CancelTimer(tostring(self.refreshTimer))
        self.refreshTimer = nil
    end
end

--- Gladdy:UpdateAttributes
-- Update button attributes
function Gladdy:UpdateAttributes(unit)
    local button = self.buttons[unit]
    if (not button) then return end

    for index, click in pairs(self.db.attributes) do
        local attr = click.modifier .. "macrotext" .. click.button
        local text
        
        if (click.action == "target") then
            text = "/targetexact " .. button.name
        elseif (click.action == "focus") then
            text = "/targetexact " .. button.name .. "\n/focus\n/targetlasttarget"
        elseif (click.action == "spell") then
            text = "/targetexact " .. button.name .. "\n/cast " .. button.spell .. "\n/targetlasttarget"
        elseif (click.action == "macro") then
            text = button.spell:gsub("*name*", button.name)
        end
        
        if (text) then
            button.secure:SetAttribute(attr, text)
        end
    end
end

--- Gladdy:Sync
-- Synchronize data with party
function Gladdy:Sync()
    for unit, button in pairs(self.buttons) do
        if (button.GUID ~= nil) then
            if (not button._sh or not button._sm or button._sh ~= button._ch or button._sm ~= button._cm) then
                local output = ("%s,%s,%s,%s,%s,%s,%d,%d,%d,%d,%d"):format(
                    button.name, button.GUID, button.class, button.classL, button.raceL, button.spec,
                    button._ch, button._mh, button._cm, button._mm, button._pt
                )
                
                button._sh = button._ch
                button._sm = button._cm
                
                self:SendCommMessage("Gladdy2", output, "PARTY", nil, "ALERT")
            end
        end
    end
end

--- Gladdy:OnCommReceived
-- Fired on receive data from party sync
function Gladdy:OnCommReceived(prefix, message, dist, sender)
    if (prefix == "GladdyTrinketUsed" and self.db.trinketStatus) then
        for unit, button in pairs(self.buttons) do
            if (button.GUID == message) then
                self:TrinketUsed(unit)
                break
            end
        end
    elseif (prefix == "Gladdy2" and sender ~= UnitName("player")) then
        local n, g, c, cl, rl, s, ch, mh, cm, mm, pt = message:split(",")
        
        local unit = self.arenaGUID[g]
        local button = unit and self.buttons[unit] or nil
        
        if (not unit) then
            unit = "arena" .. self.currentUnit
            self.currentUnit = self.currentUnit + 1
            self.arenaGUID[g] = unit
            
            button = self.buttons[unit]
            if (not button) then return end
            
            button.name = n
            button.GUID = g
            button.class = c
            button.classL = cl
            button.raceL = rl
            button.spec = s
            
            button.text:SetText(n)
            
            if (self.db.raceText) then
                button.raceText:SetText(rl)
            end
            
            if (self.db.enemyAnnounce and not button.enemyAnnounced and n ~= UNKNOWNOBJECT) then
                self:Announce(n .. " - " .. cl, RAID_CLASS_COLORS[c])
                button.enemyAnnounced = true
            end
            
            if (s ~= "") then
                self:UpdateSpec(unit, s)
            end
            
            button.health:SetStatusBarColor(RAID_CLASS_COLORS[c].r, RAID_CLASS_COLORS[c].g, RAID_CLASS_COLORS[c].b, 1.0)
            
            button.classIcon:SetTexture("Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes")
            button.classIcon:SetTexCoord(unpack(CLASS_BUTTONS[c]))
            button.classIcon:SetAlpha(1)
            
            self:HandleCastEnd(button.castBar)
            
            button:SetAlpha(1)
        end
        
        self:HandleUpdateHealth(unit, ch, mh)
        self:HandleUpdatePower(unit, cm, mm, pt)
    end
end

--- Gladdy:UpdateNameplates
-- Update healthbars from nameplates
function Gladdy:UpdateNameplates()
    if (self.childFrames ~= WorldFrame:GetNumChildren()) then
        self.childframes = WorldFrame:GetNumChildren()
        self:UpdateBars(WorldFrame:GetChildren())
    end
end

--- Gladdy:UpdateBars
-- Register hook for healthbar
function Gladdy:UpdateBars(...)
    for i = 1, select("#", ...) do
        local healthBar = self:FindHealthBar(select(i, ...):GetChildren())
        if (healthBar) then
            self:HookScript(healthBar, "OnValueChanged")
            healthBar.GladdyHooked = true
        end
    end
end

--- Gladdy:FindHeathBar
-- Find healthbar from frames
function Gladdy:FindHealthBar(...)
    for i = 1, select("#", ...) do
        local healthBar = select(i, ...)
        
        if (healthBar and not healthBar.GladdyHooked and healthBar.GetFrameType and healthBar:GetFrameType() == "StatusBar" and not healthBar:GetName() and healthBar:IsVisible()) then
            return healthBar
        end
    end
end

--- Gladdy:OnValueChanged
-- Hook for updating health from nameplates
function Gladdy:OnValueChanged(object, value)
    if (not object:IsVisible()) then
        return
    end
    
    local name = select(5, object:GetParent():GetRegions()):GetText()
    
    for unit, button in pairs(self.buttons) do
        if (button.GUID and button.name == name and value) then
            self:HandleUpdateHealth(unit, value, button._mh)
        end
    end
end

--- Gladdy:HandleAuraGain
-- Handle aura gain
function Gladdy:HandleAuraGain(unit, auraName, duration)
    local button = self.buttons[unit]
    local aura = self.auras[auraName]
    if (not button or not aura) then return end

    if (not button.spec and aura.spec) then
        self:UpdateSpec(unit, aura.spec)
    end

    if (auraName == NS) then
        button.ns = true
    elseif (auraName == NF) then
        button.nf = true
    elseif (auraName == POM) then
        button.pom = true
    elseif (auraName == DRINK and self.db.drinkAnnounce) then
        self:Announce(L["DRINK - %s (%s)"]:format(button.name, button.classL), RAID_CLASS_COLORS[button.class])
    elseif (duration and self:AuraAnnounce(unit, auraName)) then
        self:Announce(L["AURA GAIN: %s (%s) - %s for %d sec"]:format(button.name, button.classL, auraName, duration), RAID_CLASS_COLORS[button.class])
        button.lastAura = auraName
    end
end

--- Gladdy:HandleAuraFade
-- Handle aura fade
function Gladdy:HandleAuraFade(unit, auraName)
    local button = self.buttons[unit]
    local aura = self.auras[auraName]
    if (not button or not aura) then return end

    if (button.lastAura == auraName) then
        button.lastAura = nil
    end
    
    if (auraName == NS) then
        button.ns = false
    elseif (auraName == NF) then
        button.nf = false
    elseif (auraName == POM) then
        button.pom = false
    end
end

--- Gladdy:HandleUpdateHealth
-- Handle health update
function Gladdy:HandleUpdateHealth(unit, ch, mh, event)
    local button = self.buttons[unit]
    if (not button or tonumber(ch) < 1) then return end
    
    local healthText
    local short = self.db.shortHpMana
    local percent = floor(100 * (ch / mh))
    
    if (self.db.healthActual) then
        healthText = short and ch > 9999 and ("%.1fk"):format(ch / 1000) or ch
    end
    
    if (self.db.healthMax) then
        local text = short and mh > 9999 and( "%.1fk"):format(mh / 1000) or mh
        
        if (healthText) then
            healthText = ("%s/%s"):format(healthText, text)
        else
            healthText = text
        end
    end
    
    if (self.db.healthPercentage) then
        if (healthText) then
            healthText = ("%s (%d%%)"):format(healthText, percent)
        else
            healthText = ("%d%%"):format(percent)
        end
    end
    
    button.healthText:SetText(healthText)
    button.health:SetValue(percent)
    
    button._ch = ch
    button._mh = mh
    
    if (self.db.lowHealthAnnounce and percent <= self.db.lowHealthPercentage and not button.lowHealth) then
        self:Announce(L["LOW HEALTH: %s (%s)"]:format(button.name, button.classL), RAID_CLASS_COLORS[button.class])
        button.lowHealth = true
    end
    
    if (button.lowHealth and percent > self.db.lowHealthPercentage) then
        button.lowHealth = false
    end
end

--- Gladdy:HandleUpdatePower
-- Handle power update
function Gladdy:HandleUpdatePower(unit, cm, mm, pt)
    local button = self.buttons[unit]
    if (not button) then return end
    
    local manaText
    local short = self.db.shortHpMana
    local percent = floor(100 * (cm / mm))
    
    if (pt == 0 and not self.db.manaDefault) then
        button.mana:SetStatusBarColor(self.db.manaColor.r, self.db.manaColor.g, self.db.manaColor.b, self.db.manaColor.a)
    elseif (pt == 1 and not self.db.rageDefault) then
        button.mana:SetStatusBarColor(self.db.rageColor.r, self.db.rageColor.g, self.db.rageColor.b, self.db.rageColor.a)
    elseif (pt == 3 and not self.db.energyDefault) then
        button.mana:SetStatusBarColor(self.db.energyColor.r, self.db.energyColor.g, self.db.energyColor.b, self.db.energyColor.a)
    else
        button.mana:SetStatusBarColor(POWER_COLOR[pt].r, POWER_COLOR[pt].g, POWER_COLOR[pt].b, POWER_COLOR[pt].a)
    end
    
    if (self.db.manaActual) then
        manaText = short and cm > 9999 and ("%.1fk"):format(cm / 1000) or cm
    end
    
    if (self.db.manaMax) then
        local text = short and mm > 9999 and ("%.1fk"):format(mm / 1000) or mm
        
        if (manaText) then
            manaText = ("%s/%s"):format(manaText, text)
        else
            manaText = text
        end
    end
    
    if (self.db.manaPercentage) then
        if (manaText) then
            manaText = ("%s (%d%%)"):format(manaText, percent)
        else
            manaText = ("%d%%"):format(percent)
        end
    end
    
    button.manaText:SetText(manaText)
    button.mana:SetValue(percent)
    
    button._cm = cm
    button._mm = mm
    button._pt = pt
end

--- Gladdy:HandleCastStart
-- Handle cast start
function Gladdy:HandleCastStart(unit, spell, icon, value, maxValue, rank, event)
    local button = self.buttons[unit]
    if (not button) then return end
    
    local castBar = button.castBar
    local spec = self.spells[spell]
    
    if (not button.spec and spec) then
        self:UpdateSpec(unit, spec)
    end

    castBar.isCasting = event == "cast"
    castBar.isChanneling = event == "channel"
    castBar.value = value
    castBar.maxValue = maxValue
    castBar:SetMinMaxValues(0, maxValue)
    castBar:SetValue(value)
    castBar.timeText:SetText(maxValue)
    castBar.icon:SetTexture(icon)
    
    if (rank ~= nil) then
        castBar.spellText:SetFormattedText("%s (%s)", spell, rank)
    else
        castBar.spellText:SetText(spell)
    end
    
    if (RES[spell] and self.db.resAnnounce) then
        self:Announce(L["RESURRECTING: %s (%s)"]:format(button.name, button.classL), RAID_CLASS_COLORS[button.class])
    end
end

--- Gladdy:HandleCastEnd
-- Handle cast end
function Gladdy:HandleCastEnd(castBar)
    castBar.isCasting = nil
    castBar.isChanneling = nil
    castBar.timeText:SetText("")
    castBar.spellText:SetText("")
    castBar.icon:SetTexture("")
    castBar:SetValue(0)
end

--- Gladdy:CheckAura
-- Check for aura being valid buff/debuff to override current frame's aura
function Gladdy:CheckAura(name, priority, track)
    local aura = self.auras[name]
    if (not aura) then return end
    
    if (aura.track == track and aura.priority >= priority) then
        return true
    end
end

--- Gladdy:AuraAnnounce
-- Should we announce aura?
function Gladdy:AuraAnnounce(unit, aura)
    local button = self.buttons[unit]
    if (not button) then return end
    
    if ((self.db.auraAnnounce or (self.db.auraList[aura] and self.db.auraList[aura] ~= "disabled")) and (not button.lastAura or button.lastAura ~= aura)) then
        return true
    end
end

--- Gladdy:AuraGain
-- Apply aura to frame
function Gladdy:AuraGain(auraFrame, name, icon, expTime, priority)
    auraFrame.name = name
    auraFrame.priority = priority
    auraFrame.timeLeft = expTime
    auraFrame.icon:SetTexture(icon)
    auraFrame.active = true
end

--- Gladdy:AuraFades
-- Fade aura from frame
function Gladdy:AuraFades(auraFrame)
    auraFrame.name = nil
    auraFrame.priority = nil
    auraFrame.text:SetText("")
    auraFrame.icon:SetTexture("")
    auraFrame.active = false
end

--- Gladdy:UpdateSpec
-- Update spec for unit
function Gladdy:UpdateSpec(unit, spec)
    local button = self.buttons[unit]
    if (not button or button.spec ~= "") then return end
    
    button.spec = spec
    
    if (self.db.specAnnounce) then
        self:Announce(L["SPEC DETECTED: %s - %s (%s)"]:format(button.name, spec, button.classL), RAID_CLASS_COLORS[button.class])
    end
    
    if (self.db.specText) then
        if (button.raceText:GetText()) then
            button.raceText:SetFormattedText("%s %s", spec, button.raceText:GetText())
        else
            button.raceText:SetText(spec)
        end
    end
end

--- Gladdy:UnitDeath
-- Unit died
function Gladdy:UnitDeath(unit)
    local button = self.buttons[unit]
    if (not button) then return end
    
    button.health:SetValue(0)
    button.healthText:SetText(L["DEAD"])
    button.mana:SetValue(0)
    button.manaText:SetText("0%")
    button.classIcon:SetAlpha(0.33)
    button.drCooldownFrame:Hide()
    
    self:AuraFades(button.auraFrame)
    self:HandleCastEnd(button.castBar)
end

--- Gladdy:Announce
-- Send announcement
function Gladdy:Announce(text, color, dest)
    local color = color or { r = 1, g = 1, b = 1 }
    local dest = dest or self.db.announceType
    
    if (self.throttle[text] and GetTime() <= self.throttle[text]) then
        return
    end
    
    self.throttle[text] = GetTime() + 5

    if (dest == "self") then
        self:Print(text)
    elseif (dest == "party" and (GetRealNumPartyMembers() > 0 or GetRealNumRaidMembers() > 0)) then
        SendChatMessage(text, "PARTY")
    elseif (dest == "bg") then
        SendChatMessage(text, "BATTLEGROUND")
    elseif (dest == "rw") then
        RaidNotice_AddMessage(RaidBossEmoteFrame, text, color)
    elseif (dest == "fct" and IsAddOnLoaded("Blizzard_CombatText")) then
        CombatText_AddMessage(text, COMBAT_TEXT_SCROLL_FUNCTION, color.r, color.g, color.b)
    elseif (dest == "msbt" and IsAddOnLoaded("MikScrollingBattleText")) then
        MikSBT.DisplayMessage(text, MikSBT.DISPLAYTYPE_NOTIFICATION, true, color.r * 255, color.g * 255, color.b * 255)
    elseif (dest == "sct" and IsAddOnLoaded("sct")) then
        SCT:DisplayText(text, color, true, "event", 1)
    elseif (dest == "parrot" and IsAddOnLoaded("parrot")) then
        Parrot:ShowMessage(text, "Notification", true, color.r, color.g, color.b)
    end
end

--- Gladdy:Test
-- Initialize test frame
function Gladdy:Test()
    for i = 1, self.currentBracket do
        local unit = "arena" .. i
        
        if (not self.buttons[unit]) then
            self.buttons[unit] = self:CreateButton(i)
        end
        
        local button = self.buttons[unit]
        local name, guid, spec, raceL, classL, class, ch, mh, cm, mm, pt
        
        if (i == 1) then
            name = UnitName("player")
            guid = UnitGUID("player")
            spec = ""
            raceL = UnitRace("player")
            classL, class = UnitClass("player")
            ch, mh = UnitHealth("player"), UnitHealthMax("player")
            cm, mm = UnitMana("player"), UnitManaMax("player")
            pt = UnitPowerType("player")
            
            self.arenaGUID[guid] = "arena1"
        else
            local test = self.testing[unit]
            name = L["Arena "] .. i
            guid = unit
            spec = test.spec
            raceL = test.rL
            classL, class = test.cL, test.c
            ch, mh = test.ch, test.mh
            cm, mm = test.cm, test.mm
            pt = test.pt
        end
        
        button.name = name
        button.GUID = guid
        button.spec = spec
        button.raceL = raceL
        button.classL = classL
        button.class = class

        self:DrPositionIcons(unit)
        
        if (i == 1) then
            self:UpdateAttributes("arena1")
            self:AuraGain(button.auraFrame, GetSpellInfo(45438), select(3, GetSpellInfo(45438)), 10, 20)
            self:TrinketUsed("arena1")
        end
        
        self:HandleUpdateHealth(unit, ch, mh)
        self:HandleUpdatePower(unit, cm, mm, pt)
        
        if (not self.db.classText and not self.db.specText) then
            button.classText:Hide()
        end

        if (not self.db.manaText) then
            button.manaText:Hide()
        end
        
        button.text:SetText(button.name)

        if (self.db.raceText) then
            button.raceText:SetText(button.raceL)
        end

        button.health:SetStatusBarColor(RAID_CLASS_COLORS[class].r, RAID_CLASS_COLORS[class].g, RAID_CLASS_COLORS[class].b, 1.0)
        
        button.classIcon:SetTexture("Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes")
        button.classIcon:SetTexCoord(unpack(CLASS_BUTTONS[class]))
        button.classIcon:SetAlpha(1)
        
        button:SetAlpha(1)
    end
end

--- Gladdy:DrPositionIcons
-- Update position for DR icons
function Gladdy:DrPositionIcons(unit)
    local button = self.buttons[unit]
    if (not button) then return end

    local lastFrame
    local anchor = self.db.drCooldownAnchor == "CENTER" and "" or self.db.drCooldownAnchor
    
    for i = 1, 16 do
        local frame = button.drCooldownFrame["icon" .. i]
        if (frame.active) then
            frame:ClearAllPoints()
            
            if (self.db.drCooldownPos == "RIGHT") then
                if (not lastFrame) then
                    frame:SetPoint(anchor .. "LEFT", button.drCooldownFrame)
                else
                    frame:SetPoint("LEFT", lastFrame, "RIGHT", 0, 0)
                end
            else
                if (not lastFrame) then
                    frame:SetPoint(anchor .. "RIGHT", button.drCooldownFrame)
                else
                    frame:SetPoint("RIGHT", lastFrame, "LEFT", 0, 0)
                end
            end

            lastFrame = frame
        end
    end
end

--- Gladdy:DrGain
-- Fired when unit got DR level increased
function Gladdy:DrGain(unit, spell)
    local button = self.buttons[unit]
    local aura = self.auras[spell]
    local dr = aura.dr
    if (not button or not dr) then return end
    
    if (not button.dr[dr] or button.dr[dr] >= 3) then
        button.dr[dr] = 0
    end
    
    button.dr[dr] = button.dr[dr] + 1
end

--- Gladdy:DrFades
-- Fire when unit got DR level decreased
function Gladdy:DrFades(unit, spell)
    local button = self.buttons[unit]
    local aura = self.auras[spell]
    if (not button or not aura.dr) then return end
    
    local dr = aura.dr
    local factor = button.dr[dr] or 1
    
    local diminished = DR_TIME[factor] or "1/2"
    
    for i = 1, 16 do
        local frame = button.drCooldownFrame["icon" .. i]
        
        if (not frame.active or frame.dr == dr) then
            frame.active = true
            frame.lastFactor = button.dr[dr]
            frame.dr = dr
            frame.timeLeft = 15
            frame.cooldown:SetCooldown(GetTime(), 15)
            frame.texture:SetTexture(self.auras[spell].icon)
            frame.text:SetText(diminished)
            
            self:DrPositionIcons(unit)
            
            frame:Show()
            frame:SetAlpha(1)

            frame:SetScript("OnUpdate", function(f, elapsed)
                f.timeLeft = f.timeLeft - elapsed
                f.timeLeftText:SetFormattedText("%d", f.timeLeft + 1)

                if (f.timeLeft <= 0) then
                    f.active = false
                    f.dr = nil
                    f.cooldown:Hide()
                    f:SetScript("OnUpdate", nil)
                    f:Hide()
                    f.text:SetText("")
                    f.timeLeftText:SetText("")
                    f:SetAlpha(0)

                    if (f.lastFactor == button.dr[dr]) then
                        button.dr[dr] = 0
                    end

                    self:DrPositionIcons(unit)
                end
            end)

            break
        end
    end
end

--- Gladdy:UpdateUnit
-- Update unit data
function Gladdy:UpdateUnit(uid)
    if (not self:UnitValid(uid)) then
        return
    end

    local unitGUID = UnitGUID(uid)

    if (not self.arenaGUID[unitGUID]) then
        local unit = "arena" .. self.currentUnit
        self.currentUnit = self.currentUnit + 1
        self.arenaGUID[unitGUID] = unit

        local button = self.buttons[unit]
        if (not button) then return end

        local name = UnitName(uid)
        local classL, class = UnitClass(uid)
        local raceL = UnitRace(uid)

        button.name = name
        button.GUID = unitGUID
        button.classL = classL
        button.class = class
        button.raceL = raceL

        button.text:SetText(name)

        if (self.db.raceText) then
            button.raceText:SetText(raceL)
        end

        if (self.db.enemyAnnounce and not button.enemyAnnounced and name ~= UNKNOWNOBJECT) then
            self:Announce(name .. " - " .. classL, RAID_CLASS_COLORS[class])
            button.enemyAnnounced = true
        end

        button.health:SetStatusBarColor(RAID_CLASS_COLORS[class].r, RAID_CLASS_COLORS[class].g, RAID_CLASS_COLORS[class].b, 1.0)

        button.classIcon:SetTexture("Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes")
        button.classIcon:SetTexCoord(unpack(CLASS_BUTTONS[class]))
        button.classIcon:SetAlpha(1)

        self:HandleCastEnd(button.castBar)
        button:SetAlpha(1)
    end

    self:UNIT_AURA("UNIT_AURA", uid)
    self:UNIT_HEALTH("UNIT_HEALTH", uid)
    self:UNIT_POWER("UNIT_POWER", uid)
end

--- Gladdy:UnitValid
-- Check unit for being valid enemy
function Gladdy:UnitValid(uid)
    if (UnitExists(uid) and UnitName(uid) and UnitIsPlayer(uid) and UnitCanAttack("player", uid) and not UnitIsCharmed(uid) and not UnitIsCharmed("player")) then
        return true
    end
end

--- TrinkedUpdate
-- Trinked status updater
local function TrinketUpdate(f, elapsed)
    if (f.endTime < GetTime()) then
        local button = Gladdy.buttons[f.unit]

        if (Gladdy.db.trinketUpAnnounce and button) then
            Gladdy:Announce(L["TRINKET READY: %s (%s)"]:format(button.name, button.classL), RAID_CLASS_COLORS[button.class])
        end

        f:SetScript("OnUpdate", nil)
    end
end

--- Gladdy:TrinketUsed
-- Fired on trinket usage
function Gladdy:TrinketUsed(unit)
    local button = self.buttons[unit]
    if (not button) then return end

    if (not button.trinketFrame) then
        button.trinketFrame = CreateFrame("Button", nil, button)
        button.trinketFrame.unit = unit
    end

    button.trinketFrame.endTime = GetTime() + 120
    button.trinketFrame:SetScript("OnUpdate", TrinketUpdate)

    CooldownFrame_SetTimer(button.cooldownFrame, GetTime(), 120, 1)

    self:Announce(L["TRINKET USED: %s (%s)"]:format(button.name, button.classL), RAID_CLASS_COLORS[button.class])
end